/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RasterPolygonTest.C
 *    Implementation of RasterPolygonTest.h.
 *
 * @author Tom Mehner
 * @date   Dec 15, 2022
 */

#include <boost/test/unit_test.hpp>

#include <geometry/RasterPolygon.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

struct PointCMP
{
	bool operator()(const Point2i& p1, const Point2i& p2) const
	{
		if (p1.y() < p2.y())
			return true;
		else if (p1.y() == p2.y() && p1.x() < p2.x())
			return true;
		return false;
	}
};

BOOST_AUTO_TEST_CASE(PolygonTest)
{
	Polygon2f poly2D;
	{
		Point2f A(0.0, 0.0);
		Point2f B(2.0, 0.0);
		Point2f C(2.0, 2.0);
		Point2f D(0.0, 2.0);
		poly2D.push_back(A);
		poly2D.push_back(B);
		poly2D.push_back(C);
		poly2D.push_back(D);
	}

	{
		std::set<Point2i, PointCMP> cells;
		const auto region = Rect2i(0, 0, 3, 3);
		rasterPolygon(poly2D, region, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              });
		BOOST_CHECK_EQUAL(cells.size(), 6);
	}

	{
		std::set<Point2i, PointCMP> cells;
		const auto region = Rect2i(0, 0, 3, 3);
		rasterPolygon(poly2D, region, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              },
		              2);
		BOOST_CHECK_EQUAL(cells.size(), 6);
	}

	// test region
	{
		std::set<Point2i, PointCMP> cells;
		const auto region = Rect2i(2, 2, 4, 4);
		rasterPolygon(poly2D, region, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              },
		              2);
		BOOST_CHECK_EQUAL(cells.size(), 0);
		const auto region2 = Rect2i(1, 1, 4, 4);
		rasterPolygon(poly2D, region2, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              },
		              2);
		BOOST_CHECK_EQUAL(cells.size(), 2);
		cells.clear();
		const auto region3 = Rect2i(0, 0, 1, 1);
		rasterPolygon(poly2D, region3, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              },
		              2);
		BOOST_CHECK_EQUAL(cells.size(), 1);
	}

	{
		Polygon2f polyTriangles;

		Point2f A(0.0, 0.0);
		Point2f B(2.0, 2.0);
		Point2f C(4.0, 0.0);
		Point2f D(6.0, 2.0);
		Point2f E(8.0, 0.0);
		polyTriangles.push_back(A);
		polyTriangles.push_back(B);
		polyTriangles.push_back(C);
		polyTriangles.push_back(D);
		polyTriangles.push_back(E);

		std::set<Point2i, PointCMP> cells;
		const auto region = Rect2i(0, 0, 9, 3);
		rasterPolygon(polyTriangles, region, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              });
		BOOST_CHECK_EQUAL(cells.size(), 12);
	}

	{
		Polygon2f polyOnGrid;

		const auto epsilon = std::numeric_limits<float>::epsilon();
		Point2f A(0.0, 0.5 - epsilon);
		Point2f B(0.0, 1.5 + epsilon);
		Point2f C(1.0, 1.5 + epsilon);
		Point2f D(1.0, 0.5 - epsilon);

		polyOnGrid.push_back(A);
		polyOnGrid.push_back(B);
		polyOnGrid.push_back(C);
		polyOnGrid.push_back(D);

		std::set<Point2i, PointCMP> cells;
		const auto region = Rect2i(0, 0, 3, 3);
		rasterPolygon(polyOnGrid, region, [](const Point2f& p) { return p; },
		              [&](int x, int y) {
			              auto i = cells.insert(Point2i{x, y});
			              BOOST_CHECK(i.second); // ensure each cell is touched just once
			              return false;
		              });
		BOOST_CHECK_EQUAL(cells.size(), 4);
	}
}

///////////////////////////////////////////////////////////////////////////////

// namespace
