/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GeometryTest.C
 *    Some geometry test cases.
 *
 * @author Tim Langner
 * @date   2010/10/24
 */

#include <boost/test/unit_test.hpp>
#include <serialization/XMLSerializer.h>
#include <geometry/Geometry.h>
#include <math/UniformRandomGenerator.h>


#include <serialization/Print.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( PointTest )
{
	Point2f point2f(1.0f, 2.0f);
	Point3f point3f(1.0f, 2.0f, 3.0f);
	Point<float,4> point4f;
	point4f << 1,2,3,4;

	BOOST_CHECK(point2f == Point2f(cv::Point_<float>(1.0f, 2.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("P2", point2f);
	xmls.serialize("P3", point3f);
	xmls.serialize("P4", point4f);

	xml.saveToFile("out.xml");

	Point2f point2fd;
	Point3f point3fd;
	Point<float,4> point4fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("P2", point2fd);
	xmlds.deserialize("P3", point3fd);
	xmlds.deserialize("P4", point4fd);

	BOOST_CHECK(point2f == point2fd);
	BOOST_CHECK(point3f == point3fd);
	BOOST_CHECK(point4f == point4fd);
}
///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( SizeTest )
{
	Size2f size2f(1.0f, 2.0f);
	Size3f size3f(1.0f, 2.0f, 3.0f);

	BOOST_CHECK_CLOSE(size2f.width(), 1.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size2f.height(), 2.0f, 0.00001f);

	BOOST_CHECK_CLOSE(size3f.width(), 1.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size3f.height(), 2.0f, 0.00001f);
	BOOST_CHECK_CLOSE(size3f.depth(), 3.0f, 0.00001f);

	BOOST_CHECK(size2f == Size2f(Point2f(1.0f, 2.0f)));
	BOOST_CHECK(size2f == Size2f(cv::Size_<float>(1.0f, 2.0f)));
	BOOST_CHECK(size3f == Size3f(Point3f(1.0f, 2.0f, 3.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("S2", size2f);
	xmls.serialize("S3", size3f);

	Size2f size2fd;
	Size3f size3fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("S2", size2fd);
	xmlds.deserialize("S3", size3fd);

	BOOST_CHECK(size2f == size2fd);
	BOOST_CHECK(size3f == size3fd);
}
///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RectTest )
{
	Rect2f rect2f(Point2f(1.0f, 2.0f), 1.0f, 2.0f);
	Box3f rect3f(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, 3.0f);

	BOOST_CHECK(rect2f == Rect2f(Point2f(1.0f, 2.0f), Point2f(2.0f, 4.0f)));
	BOOST_CHECK(rect2f == Rect2f(Point2f(2.0f, 4.0f), Point2f(1.0f, 2.0f)));
	BOOST_CHECK(rect2f == Rect2f(Point2f(1.0f, 2.0f), Size2f(1.0f, 2.0f)));
	BOOST_CHECK(rect2f == Rect2f(boost::geometry::model::box<Point2f>(Point2f(1.0f, 2.0f),
	                                                                  Point2f(2.0f, 4.0f))));
	BOOST_CHECK(rect2f == Rect2f(cv::Rect_<float>(1.0f, 2.0f,1.0f, 2.0f)));

	XMLDom xml;
	XMLSerializer xmls(xml);
	xmls.serialize("R2", rect2f);
	xmls.serialize("R3", rect3f);

	xml.saveToFile("Test.xml");

	Rect2f rect2fd;
	Box3f rect3fd;
	XMLDeserializer xmlds(xml);
	xmlds.deserialize("R2", rect2fd);
	xmlds.deserialize("R3", rect3fd);

	BOOST_CHECK(rect2f == rect2fd);
	BOOST_CHECK(rect3f == rect3fd);
}

BOOST_AUTO_TEST_CASE( RectContainsTest )
{
	{
		Rect2i r(0, 0, 5, 5);
		BOOST_CHECK(!r.contains(Point2i(0,0)));
		BOOST_CHECK(!r.contains(Point2i(4,0)));
		BOOST_CHECK(!r.contains(Point2i(0,4)));
		BOOST_CHECK(r.contains(Point2i(1,1)));
		BOOST_CHECK(r.contains(Point2i(4,4)));
	}

	{
		Rect2i r;
		for (int x = 10; x < 20; ++x)
			for (int y = 5; y < 15; ++y) {
				r |= Point2i(x,y);
			}

		// first and last row/column of added points are not contained!
		for (int x = 10; x < 20; ++x) {
			BOOST_CHECK(!r.contains(Point2i(x,5)));
			BOOST_CHECK(!r.contains(Point2i(x,14)));
		}

		for (int y = 5; y < 15; ++y) {
			BOOST_CHECK(!r.contains(Point2i(10,y)));
			BOOST_CHECK(!r.contains(Point2i(19,y)));
		}

		// adding extra margin of 1 makes first and last row/column of added points contained
		r.minCorner -= Point2i(1,1);
		r.maxCorner += Point2i(1,1);

		for (int x = 10; x < 20; ++x) {
			BOOST_CHECK(r.contains(Point2i(x,5)));
			BOOST_CHECK(r.contains(Point2i(x,14)));
		}

		for (int y = 5; y < 15; ++y) {
			BOOST_CHECK(r.contains(Point2i(10,y)));
			BOOST_CHECK(r.contains(Point2i(19,y)));
		}

	}
}

BOOST_AUTO_TEST_CASE( RectArithmeticsTest )
{
	Rect2i r(Point2i(1, 2), 3, 4);
	Rect2i r2 = r;
	BOOST_CHECK(r2.minCorner == Point2i(1, 2));
	BOOST_CHECK(r2.maxCorner == Point2i(4, 6));

	Point2i displace(2, 4);
	r2 += displace;
	BOOST_CHECK(r2.minCorner == Point2i(3, 6));
	BOOST_CHECK(r2.maxCorner == Point2i(6, 10));

	BOOST_CHECK(r2 == r + displace);
}

BOOST_AUTO_TEST_CASE( RectTestInvalid )
{
	Rect2f rect2f(Point2f(1.0f, 2.0f), 1.0f, 2.0f);
	Box3f rect3f(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, 3.0f);

	// recognize valid/invalid

	// points are reordered internally, so it is impossible to create an invalid rect from 2 points
	Rect2f rect2fInvalidPoints(Point2f(1.0f, 2.0f), Point2f(0.0f, 2.0f));
	BOOST_CHECK(rect2fInvalidPoints.isValid());

	Rect2f rect2fInvalidSize(Point2f(1.0f, 2.0f), -1.0f, 2.0f);
	BOOST_CHECK(!rect2fInvalidSize.isValid());

	Box3f rect3fInvalidPoints(Point3f(1.0f, 2.0f, 3.0f), Point3f(1.0f, 2.0f, 0.0f));
	BOOST_CHECK(rect3fInvalidPoints.isValid());

	Box3f rect3fInvalidSize(Point3f(1.0f, 2.0f, 3.0f), 1.0f, 2.0f, -1.0f);
	BOOST_CHECK(!rect3fInvalidSize.isValid());

	// union / intersection invalid with invalid -> invalid

	BOOST_CHECK((Rect2f::invalid() & Rect2f::invalid()) == Rect2f::invalid());
	BOOST_CHECK((Rect2f::invalid() | Rect2f::invalid()) == Rect2f::invalid());

	{
		Rect2f r(Rect2f::invalid());
		r &= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
		r |= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
	}

	BOOST_CHECK((Box3f::invalid() & Box3f::invalid()) == Box3f::invalid());
	BOOST_CHECK((Box3f::invalid() | Box3f::invalid()) == Box3f::invalid());

	{
		Box3f r(Box3f::invalid());
		r &= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
		r |= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
	}

	// intersection valid with invalid -> invalid

	BOOST_CHECK((rect2f & Rect2f::invalid()) == Rect2f::invalid());
	BOOST_CHECK((Rect2f::invalid() & rect2f) == Rect2f::invalid());

	{
		Rect2f r(rect2f);
		r &= Rect2f::invalid();
		BOOST_CHECK(r == Rect2f::invalid());
		r &= rect2f;
		BOOST_CHECK(r == Rect2f::invalid());
	}

	BOOST_CHECK((rect3f & Box3f::invalid()) == Box3f::invalid());
	BOOST_CHECK((Box3f::invalid() & rect3f) == Box3f::invalid());

	{
		Box3f r(rect3f);
		r &= Box3f::invalid();
		BOOST_CHECK(r == Box3f::invalid());
		r &= rect3f;
		BOOST_CHECK(r == Box3f::invalid());
	}

	// union valid with invalid -> valid

	BOOST_CHECK((rect2f | Rect2f::invalid()) == rect2f);
	BOOST_CHECK((Rect2f::invalid() | rect2f) == rect2f);

	{
		Rect2f r(Rect2f::invalid());
		r |= rect2f;
		BOOST_CHECK(r == rect2f);
		r |= Rect2f::invalid();
		BOOST_CHECK(r == rect2f);
	}

	BOOST_CHECK((rect3f | Box3f::invalid()) == rect3f);
	BOOST_CHECK((Box3f::invalid() | rect3f) == rect3f);

	{
		Box3f r(Box3f::invalid());
		r |= rect3f;
		BOOST_CHECK(r == rect3f);
		r |= Box3f::invalid();
		BOOST_CHECK(r == rect3f);
	}
}

template<typename T>
void testRectDiff(const Rect<T,2>& r1, const Rect<T,2>& r2, const std::list<Rect<T,2>> ref)
{

	std::list<Rect<T,2>> diff = r1-r2;

	BOOST_CHECK_EQUAL(diff.size(), ref.size());

	typename std::list<Rect<T,2>>::const_iterator cdiff = diff.begin(),
	                                              cref  = ref.begin();

	for (; cdiff != diff.end(); ++cdiff, ++cref)
		BOOST_CHECK(*cdiff == *cref);
}

BOOST_AUTO_TEST_CASE( RectDiffTest )
{
//	std::cout << "result = O shape" << std::endl;
	testRectDiff<float>(Rect2f(0.f, 0.f, 3.f, 3.f), Rect2f(1.f, 1.f, 1.f, 1.f),
	                    {Rect2f(0.f, 2.f, 2.f, 1.f), Rect2f(0.f, 0.f, 1.f, 2.f),
	                     Rect2f(1.f, 0.f, 2.f, 1.f), Rect2f(2.f, 1.f, 1.f, 2.f)});
	testRectDiff<int>(Rect2i(0, 0, 3, 3), Rect2i(1, 1, 1, 1),
	                  {Rect2i(0, 2, 2, 1), Rect2i(0, 0, 1, 2),
	                   Rect2i(1, 0, 2, 1), Rect2i(2, 1, 1, 2)});

//	std::cout << "result = L shape" << std::endl;
	testRectDiff<float>(Rect2f(0.f, 0.f, 3.f, 3.f), Rect2f(1.f, 1.f, 3.f, 3.f),
	                    {Rect2f(0.f, 0.f, 1.f, 3.f), Rect2f(1.f, 0.f, 2.f, 1.f)});
	testRectDiff<int>(Rect2i(0, 0, 3, 3), Rect2i(1, 1, 3, 3),
	                    {Rect2i(0, 0, 1, 3), Rect2i(1, 0, 2, 1)});

//	std::cout << "result = U shape" << std::endl;
	testRectDiff<float>(Rect2f(0.f, 0.f, 3.f, 3.f), Rect2f(-1.f, 1.f, 3.f, 1.f),
	                    {Rect2f(0.f, 2.f, 2.f, 1.f), Rect2f(0.f, 0.f, 3.f, 1.f),
	                     Rect2f(2.f, 1.f, 1.f, 2.f)});
	testRectDiff<int>(Rect2i(0, 0, 3, 3), Rect2i(-1, 1, 3, 1),
	                  {Rect2i(0, 2, 2, 1), Rect2i(0, 0, 3, 1),
	                   Rect2i(2, 1, 1, 2)});

//	std::cout << "result = I shape" << std::endl;
	testRectDiff<float>(Rect2f(0.f, 0.f, 3.f, 3.f), Rect2f(-1.f, -1.f, 3.f, 5.f),
	                    {Rect2f(2.f, 0.f, 1.f, 3.f)});
	testRectDiff<int>(Rect2i(0, 0, 3, 3), Rect2i(-1, -1, 3, 5),
	                    {Rect2i(2, 0, 1, 3)});

//	std::cout << "result = I I shape" << std::endl;
	testRectDiff<float>(Rect2f(0.f, 0.f, 3.f, 3.f), Rect2f(1.f, -1.f, 1.f, 5.f),
	                    {Rect2f(0.f, 0.f, 1.f, 3.f), Rect2f(2.f, 0.f, 1.f, 3.f)});
	testRectDiff<int>(Rect2i(0, 0, 3, 3), Rect2i(1, -1, 1, 5),
	                    {Rect2i(0, 0, 1, 3), Rect2i(2, 0, 1, 3)});
}

template<typename T>
void testRandomRectDiff(int samples, T min, T max)
{
	UniformRandomGenerator<T> rnd(min, max);
	for (int n=0; n<samples; ++n) {
		Rect<T,2> r1(rnd(), rnd(), abs(rnd())+(T)1, abs(rnd())+(T)1); // ensure valid, non-null rects
		Rect<T,2> r2(rnd(), rnd(), abs(rnd())+(T)1, abs(rnd())+(T)1);

		std::list<Rect<T,2>> diff = r1-r2;	

		// no intersections between any parts of diff
		typename std::list<Rect<T,2>>::const_iterator c1 = diff.begin();
		for (; c1 != diff.end(); ++c1) {
			typename std::list<Rect<T,2>>::const_iterator c2 = c1;
			c2++;
			for (; c2 != diff.end(); ++c2) {
				Rect<T,2> inter = *c1 & *c2; 
				BOOST_CHECK(!inter.isValid() || (inter.width() * inter.height() == 0));
			}
		}

		// unite r1,r2 intersection and all parts of diff --> r1
		Rect<T,2> u = r1 & r2;
		foreach (auto r, diff)
			u |= r;

		BOOST_CHECK(u == r1);

		// accumulate size of r1,r2 intersection (if valid) and all parts of diff --> size of r1
		float size = 0;			
		Rect<T,2> inter = r1 & r2;
		if (inter.isValid())
			size = inter.width()*inter.height();

		foreach (auto r, diff)
			size += r.width()*r.height();

		BOOST_CHECK_CLOSE(size, r1.width() * r1.height(), 0.0001);
	}
}

BOOST_AUTO_TEST_CASE( RectDiffRandomTest )
{
	testRandomRectDiff<float>(50000, -1000.f, 1000.f);
	testRandomRectDiff<int>(50000, -1000, 1000);
}


