/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ManifestLoadTest.h
 *    $Check if proxy is working correct if the associated library of a vacant
 *    class is loaded.$.
 *
 * @author Ronny Stricker
 * @date   2010/10/25
 */

#include <boost/test/unit_test.hpp>

#include <factory/Factory.h>
#include <factory/ManifestAgent.h>
#include <testLib/TestLib.h>
#include <classes/DiamondClassStructure.h>
#include <classes/MetaInfoClasses.h>
#include <classes/ParameterConstructorClasses.h>

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/*
 * @brief Check meta info parsing with user defined matching function.
 */
bool checkfunc( std::map<std::string, std::string> const& pMetaMap )
{
	return ( ( pMetaMap.find( "Meta" ) != pMetaMap.end() )
			&& ( pMetaMap.find( "Meta" )->second == "Test" )
			&& ( pMetaMap.find( "Meta2" ) != pMetaMap.end() )
			&& ( pMetaMap.find( "Meta2" )->second == "Test2" ) );
}

BOOST_AUTO_TEST_CASE( ClassProxyLibraryLoadTest )
{
	// load the manifest file of the depLib object
	ManifestAgent tManifestAgent;
	tManifestAgent.loadFile( "testTestDepLib.mani" );
	tManifestAgent.finalize();

	// get all derived children of LibObject (including the vacant class depLibOject)
	std::map<std::string, ClassProxy > tChildren =
			LibObject::CLASS().getDerivedClasses();

	// create a instance of every child -> to force loading of lib
	foreach( auto tClass, tChildren ) {
		Object* tObject = tClass.second.newInstance();
		delete tObject;
	}
	
	// try to create instance with old pointer to the object
	Object* tInstance = tChildren["mira::DepLibObject"].newInstance();
	delete tInstance;
}

bool compareFunc( map<string, string> const& metaInfo )
{
	map<string,string>::const_iterator it = metaInfo.find( "Meta1" );
	if ( it != metaInfo.end() && it->second == "fromC" )
		return true;
	return false;
}

BOOST_AUTO_TEST_CASE( ClassProxyInterfaceTest )
{
	// since the functions are checked in more detail for the Class class itself,
	// the test are concentrating on a working interface

	ClassProxy tCProxy = A::CLASS().getClassByIdentifier("mira::C");
	ClassProxy tDProxy = A::CLASS().getClassByIdentifier("mira::D");
	ClassProxy tMetaProxy = MetaA::CLASS().getClassByIdentifier("mira::MetaB");
	ClassProxy tMetaProxy2 = MetaA::CLASS().getClassByIdentifier("mira::MetaC");
	ClassProxy tParamProxy = ParamConst::CLASS().getClassByIdentifier("mira::ParamConstDerived");
	ClassProxy tParamProxy2 = ClassFactory::getClassByIdentifier("mira::ParamConst");
	
	BOOST_CHECK( tCProxy < tDProxy );
	
	BOOST_CHECK_EQUAL( tCProxy.getIdentifier(), "mira::C" );
	BOOST_CHECK_EQUAL( tCProxy.getName(), "C" );
	// at least it should return something
	int id = tCProxy.getTypeId();
	BOOST_CHECK_EQUAL( tCProxy.getTypename(), "mira::C" );
	
	BOOST_CHECK_EQUAL( tMetaProxy.getMetaInfo().size(), 2 );
	BOOST_CHECK_EQUAL( tMetaProxy.getMetaInfo("Meta2"), "fromB" );
	
	// check new instance interfaces
	Object* tObject = NULL;
	BOOST_REQUIRE_NO_THROW( tObject = tCProxy.newInstance() );
	BOOST_REQUIRE( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::C" );
	delete tObject;
	
	BOOST_REQUIRE_NO_THROW( tObject = tCProxy.newInstance<C>() );
	BOOST_REQUIRE( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::C" );
	delete tObject;
	
	BOOST_REQUIRE_NO_THROW( tObject = tParamProxy2.newInstance(1,5) );
	BOOST_REQUIRE( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::ParamConst" );
	delete tObject;
	
	BOOST_REQUIRE_NO_THROW( tObject = tParamProxy.newInstance( "mira::ParamConstDerived2", 1, 5 ) );
	BOOST_REQUIRE( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::ParamConstDerived2" );
	delete tObject;
	
	BOOST_REQUIRE_NO_THROW( tObject = tParamProxy2.newInstance( "mira::ParamConstDerived" ) );
	BOOST_REQUIRE( tObject != NULL );
	BOOST_CHECK_EQUAL( tObject->getClass().getIdentifier(), "mira::ParamConstDerived" );
	delete tObject;
	
	// check child info interface
	BOOST_CHECK_EQUAL( tParamProxy2.isClassRegistered( "mira::ParamConstDerived2" ), true );
	
	BOOST_CHECK( tParamProxy2.getClassByIdentifier( "mira::ParamConstDerived" ) == tParamProxy );
	
	vector<ClassProxy> tVector = tMetaProxy.getClassByMeta("Meta1", "fromC");
	BOOST_CHECK_EQUAL( tVector.size(), 1 );
	BOOST_CHECK( tVector[0] == tMetaProxy2 );
	
	tVector = tMetaProxy.getClassByMeta( compareFunc );
	BOOST_CHECK_EQUAL( tVector.size(), 1 );
    BOOST_CHECK( tVector[0] == tMetaProxy2 );
	
	map<string,ClassProxy> tMap = tParamProxy2.getDerivedClasses();
	BOOST_CHECK_EQUAL( tMap.size(), 2 );
	BOOST_CHECK( tMap["mira::ParamConstDerived"] == tParamProxy );
	
	tMap = tParamProxy.getDirectParents();
	BOOST_CHECK_EQUAL( tMap.size(), 1 );
	BOOST_CHECK( tMap["mira::ParamConst"] == tParamProxy2 );
	
	BOOST_CHECK_EQUAL( tParamProxy2.isBaseOf( &ParamConstDerived::CLASS() ), true );
	BOOST_CHECK_EQUAL( tParamProxy2.isBaseOf( tParamProxy ), true );
	
	BOOST_CHECK_EQUAL( tParamProxy2.isDerivedFrom( &ParamConstDerived::CLASS() ), false );
	BOOST_CHECK_EQUAL( tParamProxy2.isDerivedFrom( tParamProxy ), false );
	BOOST_CHECK_EQUAL( ParamConst::CLASS().getClassByIdentifier("mira::ParamConstDerived2").isDirectlyDerivedFrom( tParamProxy2 ), false );
	BOOST_CHECK_EQUAL( tParamProxy.isDirectlyDerivedFrom( tParamProxy2 ), true );
	
	BOOST_CHECK_EQUAL( tParamProxy.isAbstract(), false );
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
