/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Builtins.C
 *    Impl xml/macros/Builtins.h
 *
 * @author Adrian Kriegel
 * @date   Fri Jan 17 2025
 */

#include "xml/macros/Utils.h"
#include <xml/macros/Builtins.h>

#include <iterator>

#include <serialization/IgnoreMissing.h>
#include <serialization/XMLSerializer.h>
#include <utils/MakeString.h>
#include <xml/XMLDomReflect.h>

#include <xml/macros/UserDefinedMacro.h>
#include <xml/macros/XMLMacroProcessor.h>

namespace mira::xmlmacros {

XMLNode DefineMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	Definition definition;

	XMLDeserializer deserializer(node);

	deserializer.deserializeFromNode("root", definition);

	auto macro = std::make_shared<UserDefinedMacro>(std::move(definition), processor.getCurrentScopePtr());

	// By default, each macro is define in the "macro" namespace.
	// Not moving the macro ptr as otherwise we'd have to copy the name beforehand.
	processor.getCurrentScope().define(processor.makeIdentifier(macro->getDefinition().name), macro);

	return node.remove();
}

XMLNode ForMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	if (std::distance(node.cbegin(), node.cend()) != 2) {
		MIRA_THROW(XInvalidConfig, "No range or body specified for 'For' macro.");
	}

	const auto& iteratorName = *node.cbegin();

	CopyableXMLDom rangeDom;
	std::vector<CopyableXMLDom> range;
	CopyableXMLDom bodyDom;

	{
		XMLDeserializer deserializer(node);
		deserializer.deserialize(iteratorName, rangeDom);

		processor.processInPlace(rangeDom.root());
		deserializer.deserialize("Do", bodyDom);
	}

	{
		XMLDeserializer deserializer(rangeDom.root());
		deserializer.deserializeFromNode("root", range);
	}

	const auto prevScope = processor.getCurrentScopePtr();

	auto bodyScope = std::make_shared<Scope>(Scope{prevScope, {}});

	auto& definitions = bodyScope->namespaces[XMLMacroProcessor::NAMESPACE.prefix];

	for (const auto& it : range) {
		definitions[iteratorName] = std::make_shared<XMLValueMacro>(bodyScope, it);

		processor.swapScope(bodyScope);

		// Make a copy of the body as it is processed in-place.
		CopyableXMLDom bodyCopy(bodyDom);
		processor.processInPlace(bodyCopy.root());

		// Create a tempoarary marker to spread the result of the body into.
		auto marker = node.parent().add_child(XMLDom().root());
		marker.insert_content_after("\n");
		spreadNode(marker, bodyCopy.root());
	}

	// Reset the current scope.
	processor.swapScope(prevScope);

	return node.remove();
}

XMLNode SplitMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	// copy content into temporary node so that variables can be resolved
	XMLDom tmp;
	
	auto content = tmp.root().add_child("marker");

	const auto contentAttr = node.find_attribute("Words");

	// use list of words from "Words" attribute if specified
	if (contentAttr != node.attribute_end()) {
		tmp.root().add_content(contentAttr.value());
	}

	// using spreadNode() instead of copying content to allow macro resolution
	// inside the node
	spreadNode(content, node, false);

	// resolve all variables
	processor.processInPlace(tmp.root());

	std::string text;

	for (auto it = tmp.root().content_begin(); it != tmp.root().content_end(); ++it) {
		text += *it;
	}

	// iterate over space-separated words

	std::istringstream iss(text);
	std::string word;

	while (iss >> word) {
		// turn each word into <item>word</item>
		auto item = node.insert_before("item");
		item.add_content(std::move(word));
		item.insert_content_after("\n");
	}

	return node.remove();
}

XMLNode ZipMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	XMLDom tmp;
	spreadNode(tmp.root().add_child("dummy"), node);
	processor.processInPlace(tmp.root());

	ZipBody body;
	XMLDeserializer deserializer(tmp.root());
	deserializer.deserializeFromNode("root", body);

	if (body.elements.empty()) {
		MIRA_THROW(XInvalidConfig, "No elements specified for <macro:Zip>.");
	}

	const auto size = body.elements.at(0).items.size();

	for (auto& elem : body.elements) {

		if (elem.items.size() != size) {
			MIRA_THROW(XInvalidConfig, "All elements in <macro:Zip> must have the same number of items.");
		}
	}

	for (uint i = 0; i < size; ++i) {
	
		auto item = node.insert_before("item");

		for (auto& elem : body.elements) {
			auto child = item.add_child(elem.items[i].root());
		
			child.setName(elem.name);
		}
	
	}

	return node.remove();
}

XMLNode ImportInlineMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	auto fromAttr = node.find_attribute("From");

	const auto orignalName = std::get<0>(XMLMacroProcessor::getIdentifier(node)).name;

	const std::string newName = createAnonymousName(orignalName);

	// Insert an <include /> node before the macro call.
	auto include = node.insert_before("include");
	include.add_attribute("file", fromAttr.value());

	include.add_attribute("select", std::string(MakeString() << DefineMacro::PREFIX << orignalName));
	include.add_attribute("as", std::string(MakeString() << DefineMacro::PREFIX << newName));
	include.add_attribute("macroNamespace", XMLMacroProcessor::NAMESPACE.prefix);
	include.setUri(node.uri());

	node.setName(newName);
	node.remove_attribute(fromAttr);

	// Tell the XMLDomPreprocessor to process the include node next, which is followed by `node`.
	return include;
}

std::string DefineMacro::extractName(XMLDom::const_iterator node)
{
	return std::get<0>(XMLMacroProcessor::getIdentifier(node)).name.substr(strlen(PREFIX));
}

XMLNode PrintXMLMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	for (auto it = node.begin(); it != node.end(); ++it) {
		it = processor.processInPlace(it);
	}

	XMLDom dom;

	XMLDeserializer deserializer(node);
	deserializer.deserializeFromNode("root", dom);

	std::cout << "PrintXML: \n" << dom.saveToString() << std::endl;

	return node.remove();
}

} // namespace mira::xmlmacros
