/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TimeHumanReadable.C
 *    Implementation of TimeHumanReadable.h
 *
 * @author Christof Schröter
 * @date   2024/09/19
 */

#include <utils/TimeHumanReadable.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void reflect(PropertySerializer& r, DateSerializedAsYMD& date)
{
	r.property("Year",
	           getter<int>([&date]{ return date.year(); }),
	           setter<int>([&date](int y){ date = DateSerializedAsYMD(y, date.month(), date.day()); }),
	           "Year");
	r.property("Month",
	           getter<int>([&date]{ return date.month(); }),
	           setter<int>([&date](int m){ date = DateSerializedAsYMD(date.year(), m, date.day()); }),
	           "Month", PropertyHints::limits(1, 12));
	r.property("Day",
	           getter<int>([&date]{ return date.day(); }),
	           setter<int>([&date](int d){ date = DateSerializedAsYMD(date.year(), date.month(), d); }),
	           "Day", PropertyHints::limits(1, 31));
	r.roproperty("Weekday",
	             getter<std::string>([&date]{
	               return boost::lexical_cast<std::string>(date.day_of_week());
	             }),
	             "Day of week");
}

///////////////////////////////////////////////////////////////////////////////

template<typename THR, typename T>
Getter<THR> humanReadableGetterTempl(const T& t)
{
	return getter<THR>([&]{return THR(t);});
}

template<typename THR, typename T>
Setter<THR> humanReadableSetterTempl(T& t)
{
	return setter<THR>([&](const THR& hr){t = hr;});
}

template<typename THR, typename T>
Accessor<Getter<THR>, Setter<THR>>
humanReadableAccessorTempl(T& t) {
	return makeAccessor(humanReadableGetterTempl<THR>(t), humanReadableSetterTempl<THR>(t));
}

///////////////////////////////////////////////////////////////////////////////

Getter<DateSerializedAsIsoString> humanReadableGetter(const Date& date)
{
	return humanReadableGetterTempl<DateSerializedAsIsoString>(date);
}

Setter<DateSerializedAsIsoString> humanReadableSetter(Date& date)
{
	return humanReadableSetterTempl<DateSerializedAsIsoString>(date);
}

Accessor<Getter<DateSerializedAsIsoString>, Setter<DateSerializedAsIsoString>>
humanReadableAccessor(Date& date)
{
	return humanReadableAccessorTempl<DateSerializedAsIsoString>(date);
}

Getter<DateSerializedAsYMD> humanReadableGetter(const Date& date, TimeTypesSerializedAsElements)
{
	return humanReadableGetterTempl<DateSerializedAsYMD>(date);
}

Setter<DateSerializedAsYMD> humanReadableSetter(Date& date, TimeTypesSerializedAsElements)
{
	return humanReadableSetterTempl<DateSerializedAsYMD>(date);
}

Accessor<Getter<DateSerializedAsYMD>, Setter<DateSerializedAsYMD>>
humanReadableAccessor(Date& date, TimeTypesSerializedAsElements)
{
	return humanReadableAccessorTempl<DateSerializedAsYMD>(date);
}


Getter<DurationSerializedAsIsoString> humanReadableGetter(const Duration& duration)
{
	return humanReadableGetterTempl<DurationSerializedAsIsoString>(duration);
}

Setter<DurationSerializedAsIsoString> humanReadableSetter(Duration& duration)
{
	return humanReadableSetterTempl<DurationSerializedAsIsoString>(duration);
}

Accessor<Getter<DurationSerializedAsIsoString>, Setter<DurationSerializedAsIsoString>>
humanReadableAccessor(Duration& duration)
{
	return humanReadableAccessorTempl<DurationSerializedAsIsoString>(duration);
}

Getter<DurationSerializedAsHMS> humanReadableGetter(const Duration& duration, TimeTypesSerializedAsElements)
{
	return humanReadableGetterTempl<DurationSerializedAsHMS>(duration);
}

Setter<DurationSerializedAsHMS> humanReadableSetter(Duration& duration, TimeTypesSerializedAsElements)
{
	return humanReadableSetterTempl<DurationSerializedAsHMS>(duration);
}

Accessor<Getter<DurationSerializedAsHMS>, Setter<DurationSerializedAsHMS>>
humanReadableAccessor(Duration& duration, TimeTypesSerializedAsElements)
{
	return humanReadableAccessorTempl<DurationSerializedAsHMS>(duration);
}


Getter<TimeSerializedAsIsoString> humanReadableGetter(const Time& time)
{
	return humanReadableGetterTempl<TimeSerializedAsIsoString>(time);
}

Setter<TimeSerializedAsIsoString> humanReadableSetter(Time& time)
{
	return humanReadableSetterTempl<TimeSerializedAsIsoString>(time);
}

Accessor<Getter<TimeSerializedAsIsoString>, Setter<TimeSerializedAsIsoString>>
humanReadableAccessor(Time& time) {
	return humanReadableAccessorTempl<TimeSerializedAsIsoString>(time);
}

Getter<TimeSerializedAsYMDHMS> humanReadableGetter(const Time& time, TimeTypesSerializedAsElements)
{
	return humanReadableGetterTempl<TimeSerializedAsYMDHMS>(time);
}

Setter<TimeSerializedAsYMDHMS> humanReadableSetter(Time& time, TimeTypesSerializedAsElements)
{
	return humanReadableSetterTempl<TimeSerializedAsYMDHMS>(time);
}

Accessor<Getter<TimeSerializedAsYMDHMS>, Setter<TimeSerializedAsYMDHMS>>
humanReadableAccessor(Time& time, TimeTypesSerializedAsElements) {
	return humanReadableAccessorTempl<TimeSerializedAsYMDHMS>(time);
}

///////////////////////////////////////////////////////////////////////////////

}
