/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Hexdump.C
 *    Implementation of Hexdump.h
 *
 * @author Erik Einhorn
 * @date   03/05/2012
 */

#include <utils/Hexdump.h>
#include <ctype.h>

namespace mira {
///////////////////////////////////////////////////////////////////////////////

void hexdump(std::ostream& stream, const uint8* bytes, std::size_t length)
{
	if(length==0)
		return;

	std::size_t p = 0;
	char line[16];
	for(p=0; p<length; ++p)
	{
		if(p%16 == 0) {
			// new line
			if(p>0) {
				// print chars of last line + linebreak
				stream << " |";
				for(int i=0; i<16; ++i)
					stream << line[i];
				stream << "|" << std::endl;
			}
			// print offset
			stream << std::hex << std::setfill('0') << std::setw(8) << (uint32)p << "  ";
		}

		uint8_t byte = bytes[p];

		// current byte in hex
		stream << std::hex << std::setfill('0') << std::setw(2) << (uint32)byte << " ";

		// add separator between 8th and 9th column
		if(p%16 == 7)
			stream << " ";

		// if byte is printable add it, otherwise replace it by '.'
		line[p%16] = isprint(byte) ? byte : '.';
	}

	// fill up with spaces and print chars of last line
	int remaining = ((p%16) ? (p%16) : 16);
	for (int i=0; i<16-remaining; ++i)
		stream << "   ";

	// add space for separator between 8th and 9th column (if not yet done)
	if(remaining<8)
		stream << " ";

	stream << " |";
	for(int i=0; i<remaining; ++i)
		stream << line[i];

	stream << "|" << std::endl;

	// reset stream
	stream << std::resetiosflags(std::ios_base::basefield);
}


///////////////////////////////////////////////////////////////////////////////
}
