/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Memory.C
 *    Implementation of Memory.h
 *
 * @author Tim Langner
 * @date   2011/11/27
 */

#include <platform/Memory.h>

#include <unistd.h>
#include <ios>
#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <boost/regex.hpp>

#include <utils/ToString.h>

#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename Handler>
void parseSmaps(std::ifstream& stream, Handler& handler)
{
	static const boost::regex headerRegex ("^([0-9A-Fa-f]+-[0-9A-Fa-f]+) +([^ ]*) +([0-9A-Fa-f]+) +([0-9A-Fa-f]+:[0-9A-Fa-f]+) +(\\d+) +(.*)$");
	static const boost::regex dataRegex ("^([^ ]+): +(\\d+) kB$");

	std::string line;
	MemoryBlockInfo block;
	bool haveBlock = false;
	while(!stream.eof()) {
		std::getline(stream, line);
		try {
			boost::smatch match;
			if (boost::regex_search(line, match, headerRegex)) {
				// found new block, send the last one to the handler
				if(haveBlock)
					handler(block);

				haveBlock=true;
				// create new block info
				block = MemoryBlockInfo();
				block.address = std::string(match[1].first, match[1].second);
				block.name = std::string(match[6].first, match[6].second);

				// anonymous blocks are located on the heap
				if(block.name.empty())
					block.name="[heap]";
			}
			else if(boost::regex_search(line, match, dataRegex)) {
				std::string key = std::string(match[1].first, match[1].second);
				std::string valueStr = std::string(match[2].first, match[2].second);
				uint64 value = fromString<uint64>(valueStr);

				if(key=="Private_Clean")
					block.totalPrivate+=value;
				else if(key=="Private_Dirty")
					block.totalPrivate+=value;

				else if(key=="Shared_Clean")
					block.totalShared+=value;
				else if(key=="Shared_Dirty")
					block.totalShared+=value;

				else if(key=="Swap")
					block.swap=value;

				else if(key=="Size")
					block.size=value;

				else if(key=="Rss")
					block.rss=value;

				else if(key=="Pss")
					block.pss=value;
			}
			// else unknown line
		} catch(...) {
			// ignore parse errors
		}
	}

	// send last block to the handler
	if(haveBlock)
		handler(block);
}

struct GetMemoryUsage {
	MemoryUsage mu;

	void operator()(const MemoryBlockInfo& block) {
		if(block.name=="[heap]")
			mu.totalHeap+=block.totalPrivate;
		else if(block.name.find("[stack")!=std::string::npos)
			mu.totalStack+=block.totalPrivate;
		mu.residentSetSize+=block.rss;
		mu.virtualSize+=block.size;
	}
};

MemoryUsage getMemoryUsage()
{
	GetMemoryUsage handler;
	std::ifstream smapsStream("/proc/self/smaps");
	parseSmaps(smapsStream, handler);
	return handler.mu;
}


struct GetDetailedMemoryBlockUsage {
	MemoryBlockInfoList list;
	void operator()(const MemoryBlockInfo& block) {
		list.push_back(block);
	}
};

MemoryBlockInfoList getDetailedMemoryBlockUsage()
{
	GetDetailedMemoryBlockUsage handler;
	std::ifstream smapsStream("/proc/self/smaps");
	parseSmaps(smapsStream, handler);

	return handler.list;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

