/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Console.C
 *
 * @author Erik Einhorn
 * @date   2011/02/20
 */

#include <platform/Console.h>
#include <platform/Environment.h>

#include <iostream>

#include <termios.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <stdio.h>

namespace mira { namespace console {

///////////////////////////////////////////////////////////////////////////////

void enableEcho(bool enable)
{
	struct termios tty;
	tcgetattr(STDIN_FILENO, &tty);
	if( !enable )
		tty.c_lflag &= ~ECHO;
	else
		tty.c_lflag |= ECHO;

	(void) tcsetattr(STDIN_FILENO, TCSANOW, &tty);
}

static const char* vt100ColorTbl[] = {
		"\033[0m",     // DEFAULT

		"\033[0;30m", // BLACK
		"\033[1;30m", // DARKGREY
		"\033[0;37m", // LIGHTGREY
		"\033[1;37m", // WHITE

		"\033[0;31m", // RED
		"\033[0;32m", // GREEN
		"\033[0;33m", // YELLOW
		"\033[0;34m", // BLUE
		"\033[0;35m", // MAGENTA
		"\033[0;36m", // CYAN

		"\033[1;31m", // LIGHTRED
		"\033[1;32m", // LIGHTGREEN
		"\033[1;33m", // LIGHTYELLOW
		"\033[1;34m", // LIGHTBLUE
		"\033[1;35m", // LIGHTMAGENTA
		"\033[1;36m"  // LIGHTCYAN
};

// list with known vt100 compatible terminals
static const char* vt100TermTbl[] = {
	"Eterm",
	"ansi",
	"color-xterm",
	"con132x25",
	"con132x30",
	"con132x43",
	"con132x60",
	"con80x25",
	"con80x28",
	"con80x30",
	"con80x43",
	"con80x50",
	"con80x60",
	"cons25",
	"console",
	"cygwin",
	"dtterm",
	"eterm-color",
	"gnome",
	"gnome-256color",
	"konsole",
	"konsole-256color",
	"kterm",
	"linux",
	"msys",
	"linux-c",
	"mach-color",
	"mlterm",
	"putty",
	"rxvt",
	"rxvt-256color",
	"rxvt-cygwin",
	"rxvt-cygwin-native",
	"rxvt-unicode",
	"rxvt-unicode-256color",
	"screen",
	"screen-256color",
	"screen-256color-bce",
	"screen-bce",
	"screen-w",
	"screen.linux",
	"vt100",
	"xterm",
	"xterm-16color",
	"xterm-256color",
	"xterm-88color",
	"xterm-color",
	"xterm-debian",
	NULL
};



static bool isaTTY(FILE* stream)
{
	return isatty(fileno(stream))==1;
}

static bool isVT100Compatible(FILE* stream)
{
	// if we are running on emacs, we are not compatible with VT100
	try {
		if(resolveEnvironmentVariable("EMACS")[0]=='t')
			return false;
	} catch(...) {}

	std::string term;
	try {
		term = resolveEnvironmentVariable("TERM");
	} catch(...) {
		return false;
	}

	// check if terminal is one of the compatible ones
	bool compatible = false;
	for(const char** names = vt100TermTbl; *names != NULL; ++names)
		if(term==*names) {
			compatible=true;
			break;
		}

	if(!compatible)
		return false;

	// seems to be compatible, now check if it is a terminal
	return isaTTY(stream);
}



void setTextColor(TextColor color, StdOutput output)
{
	// check for support of vt100 color escape sequences:

	if(output==STD_ERROR) {
		static bool isCompatible = isVT100Compatible(stderr);
		if(!isCompatible)
			return; // do not try to set color

		std::cerr << vt100ColorTbl[color];
		std::cerr.flush();
	} else {
		static bool isCompatible = isVT100Compatible(stdout);
		if(!isCompatible)
			return; // do not try to set color

		std::cout << vt100ColorTbl[color];
		std::cout.flush();
	}
}

int getWidth()
{
	const int fallbackWidth = 80;

	struct winsize w;
	if(ioctl(STDOUT_FILENO, TIOCGWINSZ, &w)<0)
		return fallbackWidth; // error, return fallback value
	return w.ws_col;
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace
