/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SignalHandler.C
 *    Implementation of SignalHandler.h.
 *
 * @author Erik Einhorn
 * @date   2010/10/07
 */

#include <map>

#include <error/SignalHandler.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

typedef map<int, boost::function<void (const IntSignal&)> > SignalToCallbackMap;
static SignalToCallbackMap sSignalCallbacks;

///////////////////////////////////////////////////////////////////////////////

void sigXXXHandler(int signal)
{
	// look up the signal handler
	SignalToCallbackMap::iterator it = sSignalCallbacks.find(signal);
	const std::size_t maxStackDepth = 16;
	const std::size_t stackStart = 3;

	// delegate to the installed handler, if we have a handler
	if(it!=sSignalCallbacks.end()) {
		CallStack stack = CallStack::backtrace(maxStackDepth, stackStart);
		IntSignal s(signal, stack);
		it->second(s);
	}
}

///////////////////////////////////////////////////////////////////////////////

void installSignalHandler(int sig, boost::function<void (const IntSignal&)> function)
{
	sSignalCallbacks[sig] = function;

	// with linux we could also use the above signal() but according to the
	// man page it is deprecated and should be replaced by sigaction()
	struct sigaction act;
	act.sa_handler = sigXXXHandler;
	act.sa_flags   = SA_RESETHAND; // make sure to call the original handler
	sigemptyset(&act.sa_mask);
	sigaction(sig, &act, NULL);
}

///////////////////////////////////////////////////////////////////////////////

}
