/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogFileSink.C
 *    Implementation of LogFileSink.h.
 *
 * @author Christian Martin
 * @date   2012/07/23
 */

#include <error/LogFileSink.h>
#include <error/LogSimpleFormatter.h>

#include <boost/format.hpp>

#include <utils/PathFinder.h>
#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

LogFileSink::LogFileSink(const std::string& logFilePrefix,
                         int logFileRotationDepth /* = 0 */,
                         bool useDailyLogFiles /* = false */,
                         Duration dailyLogFileStart /* = Duration(0, 0, 0) */) :
	LogSink(),
	mLogFilePrefix(resolvePath(Path(logFilePrefix)).replace_extension().string()), // remove file extension
	mFileRotationDepth(logFileRotationDepth),
	mDailyLogFiles(useDailyLogFiles),
	mDailyLogFileStart(dailyLogFileStart),
	mNeedRotation(true)
{
	mFormatter.reset(new LogSimpleFormatter());
	prepareFileStream();
}

LogFileSink::LogFileSink(const LogFileSink& other) :
	LogSink(other),
	mLogFilePrefix(other.mLogFilePrefix),
	mFileRotationDepth(other.mFileRotationDepth),
	mDailyLogFiles(other.mDailyLogFiles),
	mDailyLogFileStart(other.mDailyLogFileStart),
	mNeedRotation(other.mNeedRotation)
{
	prepareFileStream();
}

void LogFileSink::consume(const LogRecord& record)
{
	if (mFormatter == NULL)
		return;

	if (mDailyLogFiles) {
		// For daily log file it is necessary to check if we have to
		// create a new log file.
		boost::gregorian::date date = getDateForDailyLogFile();
		if (date != mCurrentDailyLogFileDate)
			prepareFileStream();
	}

	mStream << mFormatter->format(record) << std::endl;
	mStream.flush();
}

///////////////////////////////////////////////////////////////////////////////

void LogFileSink::prepareFileStream()
{
	std::string fileName;

	if (mDailyLogFiles) {
		// Determine the date string of the log file name
		boost::gregorian::date date = getDateForDailyLogFile();
		std::string dateStr = makeDateStr(date);

		// the current log file name
		fileName = mLogFilePrefix+"_"+dateStr+".log";
		mCurrentDailyLogFileDate = date;

		if (mFileRotationDepth == 0) {
			// no log file rotation => nothing to do
		} else
		if (mFileRotationDepth > 0) {
			// Get a list of all log files in the specified directory.
			Path parentDir = Path(mLogFilePrefix).parent_path();
			Path searchPattern = Path(Path(mLogFilePrefix).filename().string() + "*.log");
			PathVector logFiles = findFiles(parentDir, searchPattern);

			try {
				// Sort the list of log files and delete all old log files
				std::sort(logFiles.begin(), logFiles.end());
				while(logFiles.size() > (size_t)mFileRotationDepth) {
					boost::filesystem::remove(logFiles.front());
					logFiles.erase(logFiles.begin());
				}
			} catch(...) {}
		} else
		if (mFileRotationDepth < 0) {
			// unlimited log file rotation => nothing to do
		}
	} else {
		// the current log file name
		fileName = mLogFilePrefix+".log";

		if (mFileRotationDepth == 0) {
			// no log file rotation => nothing to do
		} else
		if (mFileRotationDepth > 0) {
			// log file rotation with limited number of files
			if (mNeedRotation) {
				for (int i = mFileRotationDepth-1; i >= 0; i--) {
					std::string src = mLogFilePrefix+".log."+toString<int>(i);
					if (i == 0)
						src = mLogFilePrefix+".log";
					std::string dest = mLogFilePrefix+".log."+toString<int>(i+1);
					if (boost::filesystem::exists(src))
						boost::filesystem::rename(src, dest);
				}
				mNeedRotation = false;
			}
		} else
		if (mFileRotationDepth < 0) {
			// unlimited log file rotation
			if (mNeedRotation) {
				PathVector files = findFiles(".", mLogFilePrefix+".log.*");
				for (int i = files.size(); i >= 0; i--) {
					std::string src = mLogFilePrefix+".log."+toString<int>(i);
					if (i == 0)
						src = mLogFilePrefix+".log";
					std::string dest = mLogFilePrefix+".log."+toString<int>(i+1);
					if (boost::filesystem::exists(src))
						boost::filesystem::rename(src, dest);
				}
				mNeedRotation = false;
			}
		}
	}

	// open the stream
	if (mStream.is_open())
		mStream.close();
	mStream.open(fileName, std::ofstream::out | std::ofstream::app);
}

boost::gregorian::date LogFileSink::getDateForDailyLogFile()
{
	Time localTime = Time::now().toLocal();
	Duration now = localTime.time_of_day();
	boost::gregorian::date date = localTime.date();
	if (now < mDailyLogFileStart) {
		// We are before the daily log file start time. Therefore, use the
		// date of yesterday of log file name creation.
		date = (localTime-Duration::hours(24)).date();
	}
	return date;
}

std::string LogFileSink::makeDateStr(boost::gregorian::date& date)
{
	std::stringstream ss;
	ss << boost::format("%02i-%02i-%02i") %
			(int)date.year() % (int)date.month() % (int)date.day();
	return ss.str();
}

///////////////////////////////////////////////////////////////////////////////

}
