/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathFinderConstExpr.h
 *    Compile-time path handling
 *
 * @author Christof Schröter
 * @date   2024/06/25
 */

#ifndef _MIRA_PATHFINDERCONSTEXPR_H_
#define _MIRA_PATHFINDERCONSTEXPR_H_

#include <boost/preprocessor/stringize.hpp>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Returns the MIRA path that contains the given path.
 * @param[in] path the path to be processed
 * @param[in] miraPaths the MIRA paths
 * @return the mira path that contains the given path, or "" if none
 *
 * @ingroup PathModule
 *
 * The option to provide MIRA paths as argument is useful for testing.
 * See below for an overload that does not require this.
 */
constexpr std::string_view findContainingMIRAPath(const std::string_view& path,
                                                  const std::string_view& miraPaths)
{
	auto pathDelim = '/';
	auto pathsDelim = ':';

	const auto& npos = std::string_view::npos;
	size_t pos = 0;
	while (true) {
		size_t to = miraPaths.find(pathsDelim, pos);
		if (to-pos > 1) {
			auto miraPath = miraPaths.substr(pos, to == npos ? npos : to-pos);

			// mirapath must end without trailing '/'
			// but the matching path must be exactly miraPath or continue with a '/' !
			// e.g. mirapath = /home/user/mira
			//      matches    /home/user/mira/any/path
			//      but NOT    /home/user/mira-pkg/any/path

			while (miraPath.back() == pathDelim)
				miraPath = miraPath.substr(0, miraPath.size()-1);

			if (path.compare(miraPath) == 0)
				return miraPath;
			else if (path.size() > miraPath.size()) {
				if ((path.compare(0, miraPath.size(), miraPath) == 0) && (path[miraPath.size()] == pathDelim))
					return miraPath;
			}
		}

		if (to == npos)
			break;
		else pos = to+1;
	}

	return "";
}

/**
 * As above, overload for const char* path
 *
 * @ingroup PathModule
 *
 * Note that you cannot just convert the result to a const char*,
 * as the size is needed in addition to the data pointer (matching null-termination can not be assumed)!
 */
constexpr std::string_view findContainingMIRAPath(const char* path, const char* miraPaths)
{
	return findContainingMIRAPath(std::string_view(path), std::string_view(miraPaths));
}

/**
 * With this overload the MIRA paths are not provided as argument,
 * they are expected to be defined as a compiler definition MIRA_PATHS.
 * Otherwise this will return an empty string literal "".
 *
 * @ingroup PathModule
 *
 * Note that you cannot just convert the result to a const char*,
 * as the size is needed in addition to the data pointer (matching null-termination can not be assumed)!
 */
constexpr std::string_view findContainingMIRAPath(const char* path)
{
#ifndef MIRA_PATHS
	return "";
#endif

	return findContainingMIRAPath(path, BOOST_PP_STRINGIZE(MIRA_PATHS));
}

/**
 * Removes the root part of the given path that belongs to one of MIRA paths.
 * E.g.:
 * \code
 *    chopMIRAPath('/home/user/mira/some/sub/folder')
 * \endcode
 * will return 'some/sub/folder' if '/home/user/mira' is set as MIRA path.
 * @param[in] path the path to be processed
 * @param[in] miraPaths the MIRA paths
 * @return the relative path from the containing MIRA folder to the given path,
 *         or just the given path if it is not within a MIRA path
 *
 * @ingroup PathModule
 *
 * The option to provide MIRA paths as argument is useful for testing.
 * See below for an overload that does not require this.
 */
constexpr std::string_view chopMIRAPath(const std::string_view& path,
                                        const std::string_view& miraPaths)
{
	auto pathDelim = '/';

	auto miraPath = findContainingMIRAPath(path, miraPaths);
	if (miraPath.empty())
		return path;

	auto pathInMiraPath = path.substr(miraPath.size());

	while (!pathInMiraPath.empty() && (pathInMiraPath.front() == pathDelim))
		pathInMiraPath.remove_prefix(1);

	return pathInMiraPath;
}

/**
 * As above, overload for const char* path
 *
 * @ingroup PathModule
 *
 * As this only removes prefixes from path, null-termination persists and
 * it is safe to return a const char*
 */
constexpr const char* chopMIRAPath(const char* path, const char* miraPaths)
{
	return chopMIRAPath(std::string_view(path), std::string_view(miraPaths)).data();
}

/**
 * With this overload the MIRA paths are not provided as argument,
 * they are expected to be defined as a compiler definition MIRA_PATHS.
 * Otherwise this will return the unchanged argument path.
 *
 * @ingroup PathModule
 *
 * As this only removes prefixes from path, null-termination persists and
 * it is safe to return a const char*
 */
constexpr const char* chopMIRAPath(const char* path)
{
#ifndef MIRA_PATHS
	return path;
#endif

	return chopMIRAPath(path, BOOST_PP_STRINGIZE(MIRA_PATHS));
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
