/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathFinder.h
 *    Different functions for searching files or directories.
 *
 * @author Tim Langner, Christian Martin
 * @date   2011/02/21
 */

#ifndef _MIRA_PATHFINDER_H_
#define _MIRA_PATHFINDER_H_

#include <utils/Path.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Checks if environment variable(s) for MIRA is/are defined.
 * @throws Exception if variables is/are not set
 * @ingroup PathModule
 */
void checkForMIRAPath();

/**
 * Returns vector of all MIRA project paths.
 * @ingroup PathModule
 */
PathVector getMIRAPaths();

///////////////////////////////////////////////////////////////////////////////

/**
 * Find a file in a vector of paths.
 * @param[in] paths The list of paths to search in.
 * @param[in] file  The file to find.
 *
 * If the file exists in different paths, the first occurrence will be returned.
 * A sub directory structure can be given e.g.
 * \code
 *     findFile(paths, "resource/MyResourceFolder/Image.png")
 * \endcode
 *
 * @throw XIO if no file was found
 * @return The path to the found file.
 * @ingroup PathModule
 */
Path findFile(const PathVector& paths, const Path& file);

/**
 * Same as findFile above, but using all of the paths contained in $MIRA_PATH variable.
 */
Path findProjectFile(const Path& file);

///////////////////////////////////////////////////////////////////////////////

/**
 * Finds all files in a vector of paths.
 * @param[in] paths     The list of paths to search in.
 * @param[in] file      The file (can also contain wildcards like * and ?)
 * @param[in] recursive Flag for recursive search.
 *
 * A sub directory structure can be given within file e.g.
 * \code
 *     findFiles(paths, "resource/MyResourceFolder/Image*.png").
 * \endcode
 * e.g. If paths=/usr/local/MIRA it will look for all files matching 
 * /usr/local/MIRA/resource/MyResourceFolder/Image*.png.
 *
 * If recursive=true it will search (in the above example) for all occurrences
 * of resource/MyResourceFolder/Image*.png in all sub directories of
 * /usr/local/MIRA i.e. it will also find
 * /usr/local/MIRA/AnotherFolder/SubFolder/resource/MyResourceFolder/Image1.png
 *
 * @return All the found file paths.
 * @ingroup PathModule
 */
PathVector findFiles(const PathVector& paths, const Path& file,
                                      bool recursive = false);

/**
 * Same as findFiles above, but using only one search path.
 * @ingroup PathModule
  */
PathVector findFiles(const Path& path, const Path& file,
                                      bool recursive = false);

/**
 * Same as findFiles above, but using all of the paths contained in $MIRA_PATH variable.
 * @ingroup PathModule
 */
PathVector findProjectFiles(const Path& file, bool recursive = false);

//////////////////////////////////////////////////////////////////////////////

/**
 * Find a sub directory in a vector of paths.
 * @param[in] paths The list of paths to search in.
 * @param[in] path  The sub path to find.
 *
 * A sub directory structure can be given e.g. 
 * \code
 *     findDirectory(paths, "resource/MyResourceFolder/")
 * \endcode
 *
 * @throw XIO if no directory was found
 * @return The path to the found sub directory.
 * @ingroup PathModule
 */
Path findDirectory(const PathVector& paths, const Path& path);

/**
 * Same as findDirectory above, but using all of the paths contained in $MIRA_PATH variable.
 * @ingroup PathModule
 */
Path findProjectDirectory(const Path& path);

/**
 * Returns the MIRA path that contains the given path.
 * @param[in] path the path to be processed
 * @return the mira path that contains the given path
 * @throws XLogical if path is not located in one of the MIRA paths
 * @ingroup PathModule
 */
Path findContainingMIRAPath(const Path& path);

/**
 * Removes the root part of the given path that belongs to one of MIRA paths.
 * E.g.:
 * \code
 *    chopMIRAPath('/home/user/mira/some/sub/folder')
 * \endcode
 * will return 'some/sub/folder' if '/home/user/mira' is set as MIRA path.
 * @param[in] path the path to be processed
 * @return the relative path from the containing mira folder to the given path
 * @throws XLogical if path is not located in one of the MIRA paths
 * @ingroup PathModule
 */
Path chopMIRAPath(const Path& path);

///////////////////////////////////////////////////////////////////////////////

/**
 * Finds all sub directories in a vector of paths.
 * @param[in] paths     The list of paths to search in.
 * @param[in] path      The sub path to find.
 * @param[in] recursive Flag for recursive search.
 *
 * A sub directory structure can be given e.g.
 * \code
 *     findDirectories(paths, "resource/MyResourceFolder/").
 * \endcode
 * e.g. If paths=[/usr/local/MyProject, /usr/local/AnotherProject] it will look for
 * all occurrences of resource/MyResourceFolder/ in /usr/local/MyProject/ and
 * /usr/local/AnotherProject.
 * - if recursive = true it will also search for all occurrences
 *   of resource/MyResourceFolder/ in all sub directories of /usr/local/MyProject
 *   and /usr/local/AnotherProject i.e. it will also find
 *   /usr/local/MyProject/AnotherFolder/SubFolder/resource/MyResourceFolder/
 * - if recursive = false it will only find all directories that match the combination
 *   of paths[X]/path e.g. in the above example it would find exactly 2 directories
 *   /usr/local/MyProject/resource/MyResourceFolder/ and
 *   /usr/local/AnotherProject/resource/MyResourceFolder/
 *
 * @return All the found directories.
 * @ingroup PathModule
 */
PathVector findDirectories(const PathVector& paths, const Path& path,
                                            bool recursive = false);

/**
 * Same as findDirectories above, but searching only in one path.
 * @ingroup PathModule
 */
PathVector findDirectories(const Path& basePath, const Path& path,
                                            bool recursive = false);

/**
 * Same as findDirectories above, but using all of the paths contained in $MIRA_PATH
 * variable.
 * @ingroup PathModule
 */
PathVector findProjectDirectories(const Path& path,
                                                   bool recursive = false);

///////////////////////////////////////////////////////////////////////////////

/**
 * Finds a file or directory searching in all paths contained in $MIRA_PATH.
 * Tries to find the file or directory by calling:
 * - 1. findProjectFile, if no file found
 * - 2. findProjectDirectory, if no directory found
 * - 3. findProjectFiles recursively in all subdirs, if no file found
 * - 4. findProjectDirectories recursively in all subdirs
 * @throw XIO If no file or directory is found or multiple matches are found
 * @param path The path to be found
 * @return The found file or directory
 * @ingroup PathModule
 */
std::string findProjectPath(const std::string& path);

///////////////////////////////////////////////////////////////////////////////

}

#endif
