/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Path.h
 *    Functions for modifying file system paths.
 *
 * @author Tim Langner
 * @date   2010/10/01
 */

#ifndef _MIRA_PATH_H_
#define _MIRA_PATH_H_

#ifndef Q_MOC_RUN
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#endif

#include <platform/Types.h>

#include <serialization/IsTransparentSerializable.h>
#include <serialization/SplitReflect.h>
#include <serialization/PropertyHint.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Typedef of a Path (shorter version for boost::filesystem::path)
 * @ingroup PathModule
 */
typedef boost::filesystem::path Path;

/**
 * Typedef for vector of Paths
 * @ingroup PathModule
 */
typedef std::vector<Path> PathVector;

/**
 * Extracts and resolves all paths contained in an environment variable.
 * If the path is found and exists it will be added to the returned path list.
 * @param[in] env The environmental variable e.g. PATH (without variable token
 * like $ or %)
 * @return All the existing paths contained in the env variable
 * @ingroup PathModule
 */
PathVector extractPaths(const std::string& env);

/**
 * Normalizes a path: it resolves all ".." in a path.
 * For optimal results the path should be made absolute before calling this function.
 * Otherwise it is not guaranteed that all ".." are resolved 
 * e.g.
 * \code
 * normalizePath("../my/path");
 * \endcode
 * will return "../my/path" since it can not determine the parent path.
 * @note This replaces boost::filesystem::path::normalize() because it is
 * deprecated and also does not take symbolic links into account!
 * @param[in] path The path to be normalized
 * @return The normalized path
 * @ingroup PathModule
 */
Path normalizePath(const Path& path);

/**
 * Resolves a path. It will resolve environment variables, (optionally) make the path
 * absolute and finally normalized.
 * @param[in] path The path to be resolved
 * @param[in] makeAbsolute If path is relative and this flag is set the path
 * will be made absolute.
 * @return The resolved, absolute and normalized path
 * @ingroup PathModule
 * Note: for an empty input path, the relative result is empty, the absolute result is
 * the same as for '.' (current working directory).
 */
Path resolvePath(const Path& path, bool makeAbsolute=true);

/**
 * The function takes two absolute paths and returns true if the given root path
 * is part of the path given in the path argument.
 * @param[in] root path
 * @param[in] path to check
 * @ingroup PathModule
 */
bool isRootOfPath(const Path& root, const Path& path);

/**
 * Get the path of the current running application binary.
 * @return The base of the current executable
 * @ingroup PathModule
 */
Path getExecutableDirectory();

/**
 * Get the absolute path of the current user's home directory.
 *
 * Under Linux this will be the content of the HOME environment variable.
 *
 * @throw XRuntime if no home directory can be found
 * @return The path of the current user's home directory
 * @ingroup PathModule
 */
Path getHomeDirectory();

/**
 * Returns the absolute path for application-specific data. A sub-directory
 * with the specified appName will be created automatically.
 *
 * Under Linux the following directory will be returned:
 *   <code>${USER HOME}/.config/${appName}</code>
 *
 * @throw XRuntime If the application data path does not exist or cannot be
 *                 created.
 * @return The absolute path for application-specific data.
 * @ingroup PathModule
 */
Path getAppDataDirectory(const std::string& appName = "mira");

/**
 * Get the absolute path of the temporary directory.
 *
 * Under Linux the following checks are performed in the given order:
 * - If TMPDIR is set its value is returned
 * - If macro P_tmpdir is defined its value is returned
 * - If /tmp exists it is returned
 *
 * @throws XRuntime if none of the above checks yields a result
 *         (meaning: no temporary directory can be found)
 *
 * @return The path of the temporary directory
 * @ingroup PathModule
 */
Path getTempDirectory();

/**
 * Returns relative path from base to path.
 * E.g.:
 * \code
 *     relativizePath("/home/user/mira/toolboxes", "/home/user/mira")
 *     // returns toolboxes
 * \endcode
 * @ingroup PathModule
 */
Path relativizePath(const Path& path, const Path& base);

///////////////////////////////////////////////////////////////////////////////

/**
 * Nonintrusive reflect method for Path.
 */
template <typename Reflector>
void reflectRead(Reflector& r, Path& path)
{
	std::string s = path.string();
	r.delegate(s);
}

/**
 * Nonintrusive reflect method for Path.
 */
template <typename Reflector>
void reflectWrite(Reflector& r, Path& path)
{
	std::string s;
	r.delegate(s);
	path = s;
}

MIRA_SPLIT_REFLECT(Path)

template<typename SerializerTag>
class IsTransparentSerializable<Path,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

namespace PropertyHints {

/**
 * Tells the property editor that the path is for a file, and that it should
 * show a "File Open"/"File Save" dialog instead of a "Choose Path" dialog.
 * Moreover you can optionally specify filters. Only files that match the
 * given filter are shown. This filters parameter uses the same syntax as
 * Qt's QFileDialog. Multiple filters can be separated by ";;", e.g.
 * @code
 *     "Images (*.png *.xpm *.jpg);;Text files (*.txt);;XML files (*.xml)"
 * @endcode
 * The save parameter tells the property editor whether to use a "File Open"
 * or a "File Save" dialog, i.e. whether to select an existing file or a
 * location and name for a file to be created.
 */
inline PropertyHint file(const std::string& filters = std::string(),
                         bool save = false) {
	return PropertyHint("file", filters) | PropertyHint("save", toString(save));
}

}

///////////////////////////////////////////////////////////////////////////////

}

#endif
