/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NoExcept.h
 *    Compatible no-exception throwing specification
 *
 * @author Christof Schröter
 * @date   2020/04/02
 */

#ifndef _MIRA_NOEXCEPT_H_
#define _MIRA_NOEXCEPT_H_

#include <platform/Platform.h>

///////////////////////////////////////////////////////////////////////////////

// Note: noexcept and throw() is NOT exactly the same meaning!
//       See https://en.cppreference.com/w/cpp/language/noexcept_spec
//       "throw() - Non-throwing dynamic exception specification (unlike
//        noexcept(true) guarantees stack unwinding and may call std::unexpected)"
//
// However:
// - noexcept is not supported by older compilers
// - throw() is deprecated in C++ 17, removed in C++20
// --> using noexcept where supported and throw() otherwise seems the next best thing to do
//
// or just don't use it:
// https://akrzemi1.wordpress.com/2014/04/24/noexcept-what-for/

///////////////////////////////////////////////////////////////////////////////

#ifdef BOOST_NOEXCEPT_OR_NOTHROW
	#define MIRA_NOEXCEPT_OR_NOTHROW BOOST_NOEXCEPT_OR_NOTHROW
#endif

///////////////////////////////////////////////////////////////////////////////

// https://stackoverflow.com/questions/18387640/how-to-deal-with-noexcept-in-visual-studio
// https://gcc.gnu.org/gcc-4.6/changes.html

#ifndef MIRA_NOEXCEPT_OR_NOTHROW
	#if defined(MIRA_GNUC_VERSION)
		#if MIRA_GNUC_VERSION >= 40600
			#define MIRA_NOEXCEPT_OR_NOTHROW noexcept
		#else
			#define MIRA_NOEXCEPT_OR_NOTHROW throw()
		#endif
	#elif defined(_MSC_FULL_VER)
		#if _MSC_FULL_VER >= 190023026
			#define MIRA_NOEXCEPT_OR_NOTHROW noexcept
		#else
			#define MIRA_NOEXCEPT_OR_NOTHROW throw()
		#endif
	#elif defined(__clang__)
		#if __has_feature(cxx_noexcept)
			#define MIRA_NOEXCEPT_OR_NOTHROW noexcept
		#else
			#define MIRA_NOEXCEPT_OR_NOTHROW throw()
		#endif
	#else
		// for any unknown compiler just assume it supports noexcept
		#define MIRA_NOEXCEPT_OR_NOTHROW noexcept
	#endif
#endif

///////////////////////////////////////////////////////////////////////////////

#endif
