/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ThreadMonitor.h
 *    A thread monitor, which collects information about the resources
 *    of all running threads of the current process.
 *
 * @author Christian Martin
 * @date   2010/09/20
 */

#ifndef _MIRA_THREADMONITOR_H_
#define _MIRA_THREADMONITOR_H_

#include <map>
#include <vector>

#include <platform/Platform.h>
#include <utils/Singleton.h>
#include <utils/Time.h>

#include <thread/Thread.h>
#include <thread/ThreadID.h>

#include <serialization/adapters/std/vector>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief A thread monitor class.
 *
 * A thread monitor, which collects information about the resources
 * of all running threads.
 *
 * To add a new thread to the monitor, the method @ref addThisThread must
 * be called <b>within</b> the created thread. To remove a thread, the method
 * @ref removeThisThread or @ref removeThread should be called. If a thread is
 * stopped without calling removeThisThread first, it will be automatically
 * removed from the internal thread list when @ref getThreadInformation() is
 * called.
 *
 * The method @ref getThreadInformation returns a vector of all threads of
 * the current process (not only the threads which are added using
 * @ref addThisThread). For each thread, the creation time and the used CPU
 * time in user mode and kernel mode are determined. For threads which were
 * added using @ref addThisThread, the registered name is included in the
 * returned data.
 *
 * The method @ref getBasicThreadInformation can be used to query only the
 * serializable subset of thread information (i.e. excluding some internal
 * thread reference data).
 *
 * Example:
 * \code
 *
 *     void processThread()
 *     {
 *         ThreadMonitor::instance().addThisThread("MyThreadName");
 *         while (!boost::this_thread::interruption_requested())
 *         {
 *             ...
 *         }
 *         ThreadMonitor::instance().removeThisThread();
 *     }
 *
 *     int main(int pArgc, char* pArgv[])
 *     {
 *         ...
 *         boost::thread tMyThread(boost::bind(&processThread));
 *         ...
 *         ThreadMonitor::ThreadInfoVector tInfos =
 *             ThreadMonitor::instance().getThreadInformation();
 *     }
 * \endcode
 * @ingroup ThreadModule
 */
class ThreadMonitor :
	public LazySingleton<ThreadMonitor>
{
public:
	/** @name Constructor and destructor */
	//@{

	/// Default constructor.
	ThreadMonitor();

	/// The destructor.
	virtual ~ThreadMonitor();

	//@}

public:
	/** @name Add and remove threads. */
	//@{

	/**
	 * @brief Add the <b>calling thread</b> to the thread monitor.
	 * @param[in] pName The name of the thread.
	 */
	void addThisThread(const std::string& pName);

	/**
	 * @brief Remove the calling thread from the monitor.
	 */
	void removeThisThread();

	/**
	 * @brief Remove a thread from the monitor.
	 * @param[in] pThread The thread, which should be removed.
	 */
	void removeThread(const boost::thread& pThread);

	/**
	 * @brief Remove a thread from the monitor.
	 * @param[in] pThreadID The boost thread ID.
	 */
	void removeThread(const boost::thread::id& pThreadID);

	/**
	 * @brief Remove a thread from the monitor.
	 * @param[in] pThreadID The native thread ID.
	 */
	void removeThread(ThreadID pThreadID);

	//@}

public:
	/** @name Thread information */
	//@{

	/// Serializable thread id
	struct BasicThreadIdentity
	{
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("ID", id, "The native thread id (number)");
			r.property("Name", name, "The thread's name");
		}

		ThreadID          id;   ///< Native thread ID (not boost::thread::native_handle_type!)
		std::string       name; ///< Name of the thread
	};

	/**
	 *  Extend basic thread id by boost thread id (not serializable).
	 *  Only for local storage.
	 */
	struct ThreadIdentity : public BasicThreadIdentity
	{
		boost::thread::id bid;  ///< Boost thread ID
	};

	/// Runtime thread info
	struct ThreadInfoBase
	{
		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("CreateTime", create_time, "Thread creation time, UTC");
			r.property("UserTime", user_time, "Time spent in user mode");
			r.property("KernelTime", kernel_time, "Time spent in kernel mode");
		}

		Time      create_time;    /**< @brief Thread creation time in UTC. */
		Duration  user_time;      /**< @brief The time amount in user-mode. */
		Duration  kernel_time;    /**< @brief The time amount in kernel-mode. */
	};

	/// Combine thread identity (template param) and runtime info
	template<typename TID>
	struct ThreadInfoTempl : public ThreadInfoBase
	{
		ThreadInfoTempl() = default;

		/// implement conversion between different identity types
		template<typename OtherTID>
		ThreadInfoTempl(const ThreadInfoTempl<OtherTID>& other)
			: ThreadInfoBase(other), known(other.known), identity(other.identity)
		{}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("Known", known, "Thread is registered to the ThreadMonitor?");
			r.property("Identity", identity, "Thread identity info");
			r.template reflectBase<ThreadInfoBase>(*this);
		}

		bool      known = false;  /**< @brief Indicates whether this thread is known by the ThreadMonitor.
		                                      If true, @ref identity data in this object is valid. */
		TID       identity;       /**< @brief Thread identity data. */
	};

	/// Basic (serializable) thread info.
	using BasicThreadInfo = ThreadInfoTempl<BasicThreadIdentity>;
	/// Vector of serializable thread info.
	using BasicThreadInfoVector = std::vector<BasicThreadInfo>;

	/// Thread info including local thread id data
	using ThreadInfo = ThreadInfoTempl<ThreadIdentity>;
	/// Vector of local id thread data
	using ThreadInfoVector = std::vector<ThreadInfo>;

public:
	/**
	 * @brief Collect information about all threads of the current process.
	 * @return Vector of thread info
	 */
	ThreadInfoVector getThreadInformation();

	/**
	 * @brief Collect basic (serializable) information about
	 *        all threads of the current process.
	 * @return Vector of basic thread info
	 */
	BasicThreadInfoVector getBasicThreadInformation();

	//@}

private:
	typedef std::map<ThreadID, ThreadIdentity> ThreadMap;

	boost::mutex mThreadsMutex;
	ThreadMap mThreads;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
