/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file core.hpp
 *    Non-intrusive reflection for OpenCV core types.
 *
 * @author Christoph Weinrich, Christof Schröter
 * @date   2011/05/18
 */

#ifndef _MIRA_CORE_HPP_
#define _MIRA_CORE_HPP_

#include <opencv2/core/core.hpp>
#include <serialization/BinarySerializer.h>
#include <serialization/NoGenericReflect.h>

#include <serialization/adapters/std/vector>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Size.
 * This methods implements the reflection of width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Size& d) {
	r.property("Width", d.width, "Size width");
	r.property("Height", d.height, "Size height");
}

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Point.
 * This methods implements the reflection of width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Point& d) {
	r.property("X", d.x, "Point x");
	r.property("Y", d.y, "Point y");
}

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for serialization of cv::Rect.
 * This methods implements the reflection of x, y, width, height
 */
template<typename Reflector>
void reflect(Reflector& r, cv::Rect& d) {
	r.property("X", d.x, "Rect x");
	r.property("Y", d.y, "Rect y");
	r.property("Width", d.width, "Rect width");
	r.property("Height", d.height, "Rect height");
}

///////////////////////////////////////////////////////////////////////////////

/** @brief Reflect method for binaryStream serialization.
 * This methods implements the reflection of width, height, type,
 * and the actual image data serialization.
 */
template<typename Derived>
void reflect(BinarySerializer<Derived>& r, cv::Mat& mat) {
	// serialization is only for 2 dimensional matrices until yet
	assert(mat.dims == 2);
	r.member("Cols", mat.cols, "The width of the image");
	r.member("Rows", mat.rows, "The height of the image");
	int type = mat.type();
	r.member("Type", type, "The data type including number of channels");

	std::size_t bytesPerLine = mat.elemSize() * mat.cols;

	// serialize the image data line by line
	const uchar* data = mat.data;
	for (int i = 0; i < mat.rows; ++i, data += mat.step)
		r.write(data, bytesPerLine);
}

/** @brief Reflect method for binaryStream deserialization.
 * This methods implements the reflection for width, height, type,
 * and the image data deserialization.
 */
template<typename Derived>
void reflect(BinaryDeserializer<Derived>& r, cv::Mat& mat) {
	int cols, rows, type;
	// serialization is only for 2 dimensional matrices until yet
	r.member("Cols", cols, "The width of the image");
	r.member("Rows", rows, "The height of the image");
	r.member("Type", type, "The data type including number of channels");

	// create the empty image
	mat.create(rows, cols, type);

	std::size_t bytesPerLine = mat.elemSize() * mat.cols;

	// and deserialize the image line by line
	uchar* data = mat.data;
	for (int i = 0; i < mat.rows; ++i, data += mat.step)
		r.read(data, bytesPerLine);
}

template <typename Reflector>
void reflect(ReflectorInterface<Reflector>& r, cv::Mat& mat)
{
	splitReflect(r, mat);
}

template<typename T, typename Reflector>
void reflectReadMat(Reflector& r, cv::Mat& mat, const std::string& type) {
	std::vector<std::vector<std::vector<T>>> vec(mat.rows);

	for (int r = 0; r < mat.rows; ++r) {
		vec[r].resize(mat.cols);
		for (int c = 0; c < mat.cols; ++c) {
			vec[r][c].resize(mat.channels());
			for (int ch = 0; ch < mat.channels(); ++ch)
				vec[r][c][ch] = *(mat.ptr<T>(r,c)+ch);
		}
	}
	r.property("Type", type, "Matrix element type", PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);
	r.property("Data", vec, "Matrix elements", PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);
}

template<typename Reflector>
void reflectRead(Reflector& r, cv::Mat& mat) {

	int type = mat.depth();
	switch(type) {
		case CV_8U  : reflectReadMat<uchar>(r, mat, "8U"); break;
		case CV_8S  : reflectReadMat<char>(r, mat, "8S"); break;
		case CV_16U : reflectReadMat<ushort>(r, mat, "16U"); break;
		case CV_16S : reflectReadMat<short>(r, mat, "16S"); break;
		case CV_32S : reflectReadMat<int>(r, mat, "32S"); break;
		case CV_32F : reflectReadMat<float>(r, mat, "32F"); break;
		case CV_64F : reflectReadMat<double>(r, mat, "64F"); break;
		default: MIRA_THROW(XInvalidParameter,
		                    "cv::Mat reflection not defined for user-defined element type")
	}
}

template<typename T, typename Reflector>
void reflectWriteMat(Reflector& r, cv::Mat& mat, int type) {
	std::vector<std::vector<std::vector<T>>> vec;
	r.property("Data", vec, "Matrix elements", PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);

	if ((vec.size() == 0) || (vec[0].size() == 0) || (vec[0][0].size() == 0))
		MIRA_THROW(XInvalidParameter,
		           "cv::Mat deserialization expects non-empty matrix (rows, columns and channels > 0)")

	switch(type) {
		case CV_8U   : mat.create(vec.size(), vec[0].size(), CV_8UC(vec[0][0].size())); break;
		case CV_8S   : mat.create(vec.size(), vec[0].size(), CV_8SC(vec[0][0].size())); break;
		case CV_16U  : mat.create(vec.size(), vec[0].size(), CV_16UC(vec[0][0].size())); break;
		case CV_16S  : mat.create(vec.size(), vec[0].size(), CV_16SC(vec[0][0].size())); break;
		case CV_32S  : mat.create(vec.size(), vec[0].size(), CV_32SC(vec[0][0].size())); break;
		case CV_32F  : mat.create(vec.size(), vec[0].size(), CV_32FC(vec[0][0].size())); break;
		case CV_64F  : mat.create(vec.size(), vec[0].size(), CV_64FC(vec[0][0].size())); break;
		default: MIRA_THROW(XInvalidParameter,
		                    "cv::Mat reflection not defined for user-defined element type")
	}

	for (int r = 0; r < mat.rows; ++r) {
		if ((int)vec[r].size() != mat.cols)
			MIRA_THROW(XInvalidParameter,
			          "All rows must have equal number of elements in cv::Mat deserialization " <<
			          "(row 0 = " << vec[0].size() << " vs. row " << r << " = " << vec[r].size())
		for (int c = 0; c < mat.cols; ++c) {
			if ((int)vec[r][c].size() != mat.channels())
				MIRA_THROW(XInvalidParameter,
				          "All elements must have equal number of channels in cv::Mat deserialization " <<
				          "((0,0) = " << vec[0][0].size() << " vs. (" << r << "," << c << ")" << " = " << vec[r][c].size())
			for (int ch = 0; ch < mat.channels(); ++ch)
				*(mat.ptr<T>(r,c)+ch) = vec[r][c][ch];
		}
	}
}

template<typename Reflector>
void reflectWrite(Reflector& r, cv::Mat& mat) {

	std::string type;
	r.property("Type", type, "Matrix element type", PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);

	if (type == "8U")
		reflectWriteMat<uchar>(r, mat, CV_8U);
	else if (type == "8S")
		reflectWriteMat<char>(r, mat, CV_8S);
	else if (type == "16U")
		reflectWriteMat<ushort>(r, mat, CV_16U);
	else if (type == "16S")
		reflectWriteMat<short>(r, mat, CV_16S);
	else if (type == "32S")
		reflectWriteMat<int>(r, mat, CV_32S);
	else if (type == "32F")
		reflectWriteMat<float>(r, mat, CV_32F);
	else if (type == "64F")
		reflectWriteMat<double>(r, mat, CV_64F);
	else
		MIRA_THROW(XInvalidParameter,
		           "Found unknown or unsupported type '" << type << "' in cv::Mat deserialization");
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
