/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SplitReflect.h
 *    Provides MIRA_SPLIT_REFLECT macros.
 *
 * @author Erik Einhorn
 * @date   2010/09/23
 */

#ifndef _MIRA_SPLITREFLECT_H_
#define _MIRA_SPLITREFLECT_H_

#include <platform/Typename.h>
#include <serialization/ReflectorMacros.h>

namespace mira {

template <typename Derived>
class ReflectorInterface;

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
// default
template<typename DerivedReflector, typename T>
void reflectRead(mira::ReflectorInterface<DerivedReflector>& r, T& v)
{
	// #########################################################################
	// If you get a compiler error here, you used the splitReflect() method
	// or the MIRA_SPLIT_REFLECT macro, but forgot to specialize and implement
	// the reflectRead() and reflectWrite() methods for your type. Additionally,
	// make sure, that these functions are placed within the namespace "mira".
	// #########################################################################
	static_assert(sizeof(T)==0, "You must implement reflectRead() "
	                            "if you use the MIRA_SPLIT_REFLECT macro");
}

template<typename DerivedReflector, typename T>
void reflectWrite(mira::ReflectorInterface<DerivedReflector>& r, T& v)
{
	// #########################################################################
	// If you get a compiler error here, you used the splitReflect() method
	// or the MIRA_SPLIT_REFLECT macro, but forgot to specialize and implement
	// the reflectRead() and reflectWrite() methods for your type. Additionally,
	// make sure, that these functions are placed within the namespace "mira".
	// #########################################################################
	static_assert(sizeof(T)==0, "You must implement reflectWrite() "
	                            "if you use the MIRA_SPLIT_REFLECT macro");
}

struct GlobalReflectRead {
	template<typename Reflector, typename T>
	static void invoke(mira::ReflectorInterface<Reflector>& r, T& value) {
		auto& reflector = static_cast<Reflector&>(r);
		reflectRead(reflector, value);
	}
};

struct GlobalReflectWrite {
	template<typename Reflector, typename T>
	static void invoke(Reflector& r, T& value) {
		reflectWrite(r, value);
	}
};

template<typename GlobalReflectReadOrWrite, typename Reflector, typename T>
inline void splitReflectInvoke(Reflector& r, T& value)
{
	GlobalReflectReadOrWrite::invoke(r, value);
}

/**
 * @ingroup SerializationModule
 *
 * This method is used to split reflection into a reflectRead() and reflectWrite()
 * method. It can be called within a non-intrusive reflect() method.
 * It resolves to non-intrusive reflectRead(r, value) if the visiting Reflector is
 * reading only and non-intrusive reflectWrite(r, value) if the
 * visiting Reflector is writing only (checked at compile time).
 *
 * @see @ref SerializationPage
 */
template<typename Reflector, typename T>
inline void splitReflect(mira::ReflectorInterface<Reflector>& r, T& value)
{
	auto& reflector = static_cast<Reflector&>(r);

	/* proper use of reflect barrier enables to e.g. potentially add a version to
	   the splitting mechanism itself (--> in splitReflectMemberInvoke()) */
	static const std::string context = typeName<T>() + " splitReflect";

	if constexpr (Reflector::isReadOnly::value) {
		MIRA_REFLECT_CALL(Reflector, reflector, context.c_str(),
		                  (splitReflectInvoke<GlobalReflectRead>(reflector, value)));
	}
	else{
		MIRA_REFLECT_CALL(Reflector, reflector, context.c_str(),
		                  (splitReflectInvoke<GlobalReflectWrite>(reflector, value)));
	}
}
///@endcond

/**
 * @ingroup SerializationModule
 *
 * Macro that inserts a reflect() method consisting of just a call to
 * splitReflect() (splitting to reflectRead() / reflectWrite()).
 * In contrast to the @ref MIRA_SPLIT_REFLECT_MEMBER macro this macro can be used
 * outside of classes only, creating a non-intrusive variant of reflect() for Type.
 *
 * @see @ref SerializationPage
 */
#define MIRA_SPLIT_REFLECT(Type)                          				\
	template<typename Reflector>                            			\
	void reflect(mira::ReflectorInterface<Reflector>& r, Type& value)	\
	{                                                     				\
		splitReflect(r, value);                           				\
	}

template<typename Reflector, typename Class>
inline void splitReflectMemberInvoke(Reflector& r, Class* This)
{
	if constexpr (Reflector::isReadOnly::value) {
		This->reflectRead(r);
	}
	else {
		This->reflectWrite(r);
	}
}


///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * This method is used to split reflection into a reflectRead() and reflectWrite()
 * method. It can be called within a class member reflect() method.
 * It resolves to this->reflectRead(r) if the visiting Reflector is reading only and
 * this->reflectWrite(r) if the visiting Reflector is writing only
 * (checked at compile time).
 *
 * @see @ref SerializationPage
 */
template<typename Reflector, typename Class>
inline void splitReflectMember(mira::ReflectorInterface<Reflector>& r, Class* This)
{
	auto& reflector = static_cast<Reflector&>(r);

	/* proper use of reflect barrier enables to e.g.
	   potentially add a version to the splitting mechanism
	   itself (--> in splitReflectMemberInvoke()) */

	static const std::string context = typeName<Class>() + " splitReflectMember";

	MIRA_REFLECT_CALL(Reflector, reflector, context.c_str(),
	                  (splitReflectMemberInvoke(reflector, This)));
}

/**
 * @ingroup SerializationModule
 *
 * Macro that insert a class member reflect() method just splitting reflection
 * into a reflectRead() and reflectWrite() method. this->reflect(r) resolves to
 * this->reflectRead(r) if the visiting Reflector is reading only and
 * this->reflectWrite(r) if the visiting Reflector is writing only (checked at compile time).
 * In contrast to the @ref MIRA_SPLIT_REFLECT macro this macro can be used
 * within classes only.
 *
 * @see @ref SerializationPage
 */
#define MIRA_SPLIT_REFLECT_MEMBER															\
	template<typename Reflector>															\
	void reflect(mira::ReflectorInterface<Reflector>& r)									\
	{																						\
		Reflector& reflector = static_cast<Reflector&>(r);									\
																							\
	/* proper use of reflect barrier enables to e.g.										\
	   potentially add a version to the splitting mechanism									\
	   itself (--> in splitReflectMemberInvoke()) */										\
																							\
		typedef typename std::remove_reference<decltype(*this)>::type splitType;			\
																							\
		static const std::string context = typeName<splitType>() + " MIRA_SPLIT_REFLECT_MEMBER"; \
																							\
		MIRA_REFLECT_CALL(Reflector, reflector, context.c_str(),							\
		                  (splitReflectMemberInvoke(reflector, this)));				\
	}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_SPLITREFLECT_H_ */
