/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyTree.h
 *    Helper class for representing the tree structure of properties.
 *
 * @author Erik Einhorn
 * @date   2012/03/30
 */

#include <serialization/PropertyNode.h>
#include <serialization/adapters/std/vector>

#ifndef _MIRA_PROPERTYTREE_H_
#define _MIRA_PROPERTYTREE_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Helper class that is able to hold a complete property tree structure.
 * It allows to serialize and deserialize such a structure.
 *
 * Example for serializing a property tree:
 * \code
 * MyClass myobject;
 * // create the property tree of myobject
 * PropertyManager m;
 * m.addObject("MyObject", myobject);
 * PropertyNode* n = m.getRootNode();
 *
 * // serialize the property tree
 * PropertyTree tree(n);
 * XMLSerializer s(xmldom);
 * s.serialize("PropertyTree", tree);
 * \endcode
 *
 * From a property tree, all property nodes can be restored as follows:
 * \code
 * PropertyTree tree;
 * XMLDeserializer d(xmldom);
 * d.deserialize("PropertyTree", tree);
 * // generate all property nodes of type MyPropertyNode
 * PropertyNode* n = tree.generatePropertyNodes<MyPropertyNode>();
 * \endcode
 *
 * @ingroup SerializationModule
 */
class PropertyTree
{
private:

	/**
	 * Internal representation of a property node, including the hierarchy.
	 * PropertyNode and PropertyTree::Node are both subclasses of PropertyNodeInfo,
	 * creating one from the other copies the PropertyNodeInfo base.
	 */ 
	struct Node : public PropertyNodeInfo
	{
		Node() {}
		Node(const PropertyNodeInfo& info) : PropertyNodeInfo(info) {}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			MIRA_REFLECT_BASE(r, PropertyNodeInfo);
			r.member("Children", children, "");
		}

		std::list<Node> children;
	};

	/// creates the tree, adding child nodes of a given property node recursively
	void createChildren(Node& node, PropertyNode* pnode)
	{
		node.children.clear();
		foreach(PropertyNode* p, pnode->children())
		{
			node.children.push_back(*p);
			createChildren(node.children.back(),p);
		}
	}

	/// creates the child property nodes from the tree
	template <typename PropertyNodeType>
	void generateChildren(PropertyNodeType* pnode, const Node& node) const
	{
		assert(pnode!=NULL);
		foreach(const Node& c, node.children)
		{
			PropertyNodeType* cn = new PropertyNodeType(c);
			pnode->addChild(cn);
			generateChildren<PropertyNodeType>(cn, c);
		}
	}

public:

	/// Creates an empty tree.
	PropertyTree() {}

	/**
	 * Constructs a property tree representation from a given property node
	 * and all of its child nodes.
	 * The property nodes can be recreated from the tree using the
	 * generatePropertyNodes() methods.
	 */
	PropertyTree(PropertyNode* root)
	{
		assert(root!=NULL);
		mRoot = *root;
		createChildren(mRoot,root);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Root", mRoot, "");
	}

public:

	/**
	 * Returns the info for the root node, and therefore allows the user
	 * to create a root property node from this root node info manually.
	 */
	const PropertyNodeInfo& getRootNodeInfo() const {
		return mRoot;
	}

public:

	/**
	 * Reconstructs the property nodes from the given tree structure.
	 * Optionally, the user can specify a previously created root node
	 * (e.g. using the info obtained by getRootNodeInfo() )
	 */
	template <typename PropertyNodeType>
	PropertyNodeType* generatePropertyNodes(PropertyNodeType* root = NULL) const
	{
		// create property node, if no one was specified to be used
		if(root==NULL)
			root = new PropertyNodeType(mRoot);

		generateChildren<PropertyNodeType>(root, mRoot);
		return root;
	}

private:
	Node mRoot;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
