/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCError.h
 *    Error codes for reasons of errors/exceptions while processing
 *    an rpc call.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2011/02/21
 */

#ifndef _MIRA_RPCERROR_H_
#define _MIRA_RPCERROR_H_

#include <error/Exceptions.h>
#include <serialization/ReflectorInterface.h>
#include <serialization/adapters/boost/shared_ptr.hpp>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 * enumeration of possible reasons for errors/exceptions while performing an
 * RPC call
 */
enum RPCError
{
	RPC_INVALID_REQUEST,   ///< The request is invalid or can not be parsed.
	RPC_METHOD_NOT_FOUND,  ///< Requested method was not found
	RPC_INVALID_PARAMS,    ///< Invalid parameters were specified for the method
	RPC_EXCEPTION_IN_CALL, ///< An exception has occurred within the method that was called.
};

/**
 * @ingroup RPCModule
 * An exception that is thrown by the RPCServer if an RPC call fails.
 */
class XRPC : public XRuntime
{
	MIRA_OBJECT(XRPC)
protected:
	friend class ClassFactoryDefaultConstClassBuilder;
	XRPC() {}

public:
	XRPC(const std::string& msg, const char* file=NULL, int line=0) :
		XRuntime(msg, file, line) {}

	virtual ~XRPC() MIRA_NOEXCEPT_OR_NOTHROW {}

	template<typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, XRuntime);
		r.member("OriginalException", origException, "");
	}

	virtual void raise(bool recursive = false)
	{
		if (recursive && origException)
			origException->raise(recursive);

		throw *this;
	}

public:

	/// takes ownership of ex
	void setOrigException(SerializableException* ex) { origException.reset(ex); }
	void setOrigException(SerializableExceptionPtr ex) { origException = ex; }
	bool hasOrigException() { return (bool)origException; }
	void raiseOrigException(bool recursive = false) { origException->raise(recursive); }

protected:
	SerializableExceptionPtr origException;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
