/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SchmittTriggerHysteresis.h
 *    Dual threshold hysteresis trigger class.
 *
 * @author Tim Langner
 * @date   2011/04/09
 */

#ifndef _MIRA_SCHMITTTRIGGERHYSTERESIS_H_
#define _MIRA_SCHMITTTRIGGERHYSTERESIS_H_

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * Class realizing a Schmitt trigger. This is a dual threshold trigger.
 * It is called trigger because the output retains its value until the
 * input changes sufficiently to trigger a change.
 * When the input is higher than a certain chosen threshold, the output is high.
 * When the input is below a different (lower) chosen threshold, the output is
 * low. When the input is between the two, the output retains its value.
 * @ingroup MathModule
 */
template <typename T>
class SchmittTriggerHysteresis
{
public:
	/**
	 * State of the trigger
	 */
	enum State
	{
		UNKNOWN,
		LOW,
		HIGH
	};

	SchmittTriggerHysteresis() :
		mState(UNKNOWN)
	{}

	SchmittTriggerHysteresis(State state) :
		mState(state)
	{}

	SchmittTriggerHysteresis(const T& lower, const T& upper) :
		mState(UNKNOWN),
		mLowerThreshold(lower<=upper?lower:upper),
		mUpperThreshold(upper>=lower?upper:lower)
	{}

	SchmittTriggerHysteresis(State state, const T& lower, const T& upper) :
		mState(state),
		mLowerThreshold(lower<=upper?lower:upper),
		mUpperThreshold(upper>=lower?upper:lower)
	{}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Lower", mLowerThreshold, "Lower threshold");
		r.property("Upper", mUpperThreshold, "Upper threshold");
		r.member("State", mState, "The state", UNKNOWN);
	}

	/**
	 * Feed the next value into the trigger.
	 * Returns true if the state has changed.
	 */
	bool update(const T& value)
	{
		State oldState = mState;
		if (value < mLowerThreshold)
			mState = LOW;
		if (value > mUpperThreshold)
			mState = HIGH;
		if (mState != oldState)
			return true;
		return false;
	}

	/**
	 * Returns the state of the trigger.
	 */
	State getState() const
	{
		return mState;
	}

private:
	State mState;
	T mLowerThreshold;
	T mUpperThreshold;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
