/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Size.h
 *    This class provides templatized multidimensional sizes for
 *    multidimensional geometric objects. Here also specialization for the 2D
 *    and 3D case are provided.
 *
 * @author Tim Langner
 * @date   2010/10/24
 */

#ifndef _MIRA_SIZE_H_
#define _MIRA_SIZE_H_

#include <geometry/Point.h>

#include <utils/IsCheapToCopy.h>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Size class for defining sizes with different data types. Here the class has
 * two template parameters. T for the data type like float or int, and the
 * second for the number of dimensions of the size.
 */
template<typename T, int D>
class Size : public Eigen::Matrix<T,D,1>
{
public:
	typedef Eigen::Matrix<T,D,1> BaseType;
	typedef Point<T,D> PointType;

public:
	/// the default constructor
	Size() {}

	/** a constructor to initialize width and height with a Eigen-matrix of correct
	 dimension */
	template <typename MatrixDerived>
	Size(const Eigen::MatrixBase<MatrixDerived>& matrix) :
		BaseType(matrix) {}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Specialization for 2D with special members width() and height().
 */
template<typename T>
class Size<T,2> : public Eigen::Matrix<T,2,1>
{
public:
	typedef Eigen::Matrix<T,2,1> BaseType;
	typedef Point<T,2> PointType;

public:
	/** @name Constructors */
	//@{

	/// the default constructor
	Size() {}

	/** a constructor to initialize width and height with a Eigen-matrix of correct
	 dimension */
	template <typename MatrixDerived>
	Size(const Eigen::MatrixBase<MatrixDerived>& matrix) :
		BaseType(matrix) {}

	/// a constructor with width and height for the 2D case
	Size(T width, T height) :
		BaseType(width, height) {}

	/// a copy constructor
	explicit Size(const cv::Size_<T> size) :
		BaseType(size.width, size.height) {}

	//@}

public:
	/** @name Generic operations */
	//@{

	/// return the width
	T width() const	{
		return this->x();
	}

	/// return the height
	T height() const {
		return this->y();
	}

	/// return the width
	T& width() {
		return this->x();
	}

	/// return the height
	T& height() {
		return this->y();
	}

	template<typename Reflector>
	void reflect(Reflector& reflector)
	{
		reflector.property("Width", this->x(), "The width");
		reflector.property("Height", this->y(), "The height");
	}

	//@}

public:
	/** @name Implicit conversion operators */
	//@{

	/// converts to mira Point
	operator PointType() const
	{
		return PointType(this->x(), this->y());
	}

	/// converts to open cv size
	operator cv::Size_<T>() const
	{
		return cv::Size_<T>(this->x(), this->y());
	}

	/// converts from cv size
	Size& operator=(const cv::Size_<T>& other)
	{
		(*this)[0] = other.width;
		(*this)[1] = other.height;
		return *this;
	}

	//@}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Specialization for 3D with special members width(),height() and depth().
 */
template<typename T>
class Size<T,3> : public Eigen::Matrix<T,3,1>
{
public:
	typedef Eigen::Matrix<T,3,1> BaseType;
	typedef Point<T,3> PointType;

public:
	/** @name Constructors */
	//@{

	/// the default constructor
	Size() {}

	/** a constructor to initialize width and height with a Eigen-matrix of correct
	 dimension */
	template <typename MatrixDerived>
	Size(const Eigen::MatrixBase<MatrixDerived>& matrix) :
		BaseType(matrix) {}

	/// a constructor to initialize the size with width, height, and depth in the 3D case
	Size(T width, T height, T depth) :
		BaseType(width, height, depth) {}

	//@}

public:
	/** @name Generic operations */
	//@{

	/// return the width
	T width() const {
		return this->x();
	}

	/// return the height
	T height() const {
		return this->y();
	}

	/// return the depth
	T depth() const {
		return this->z();
	}

	/// return the width
	T& width() {
		return this->x();
	}

	/// return the height
	T& height() {
		return this->y();
	}

	/// return the depth
	T& depth() {
		return this->z();
	}

	/// the serialization definition
	template<typename Reflector>
	void reflect(Reflector& reflector)
	{
		reflector.property("Width", this->x(), "The width");
		reflector.property("Height", this->y(), "The height");
		reflector.property("Depth", this->z(), "The depth");
	}

	//@}

public:
	/** @name Implicit conversion operators */
	//@{

	/// converts to mira Point
	operator PointType() const
	{
		return PointType(this->x(), this->y(), this->z());
	}

	//@}
};

///////////////////////////////////////////////////////////////////////////////

/// The size type for 2D objects in integer precision.
typedef Size<int, 2> Size2i;

/// The size type for 2D objects in floating point precision.
typedef Size<float, 2> Size2f;

/// The size type for 2D objects in 64 bit floating point precision.
typedef Size<double, 2> Size2d;

/// The size type for 3D objects in integer precision.
typedef Size<int, 3> Size3i;

/// The size type for 3D objects in floating point precision.
typedef Size<float, 3> Size3f;

/// The size type for 3D objects in 64 bit floating point precision.
typedef Size<double, 3> Size3d;

///////////////////////////////////////////////////////////////////////////////

template <>
class IsCheapToCopy<Size2i> : public std::true_type {};

template <>
class IsCheapToCopy<Size2f> : public std::true_type {};

template <>
class IsCheapToCopy<Size2d> : public std::true_type {};

template <>
class IsCheapToCopy<Size3i> : public std::true_type {};

template <>
class IsCheapToCopy<Size3f> : public std::true_type {};

template <>
class IsCheapToCopy<Size3d> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
