/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerialPort.h
 *    Wrapper class for boost::asio::serial_port to add baudrate support
 *    for non standard baudrates.
 *
 * @author Tim Langner
 * @date   2011/06/24
 */

#ifndef _MIRA_SERIALPORT_H_
#define _MIRA_SERIALPORT_H_

#ifndef Q_MOC_RUN
#include <boost/asio/serial_port.hpp>
#endif

#include <platform/Types.h>

#define MIRA_ASIO_OPTION_STORAGE termios

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * This class acts as a wrapper to boost::asio::serial_port.
 * It provides the usage of non standard baud rates via its Baudrate class.
 * Supported baud rates are:
 * - 0
 * - 50
 * - 75
 * - 110
 * - 134
 * - 150
 * - 200
 * - 300
 * - 600
 * - 1200
 * - 1800
 * - 2400
 * - 4800
 * - 7200
 * - 9600
 * - 14400
 * - 19200
 * - 38400
 * - 57600
 * - 115200
 * - 172800
 * - 230400
 * - 288000
 * - 345600
 * - 403200
 * - 460800
 * - 500000
 * - 576000
 * - 687500
 * - 691200
 * - 768000
 * - 806400
 * - 921600
 * - 1000000
 * - 1031000
 * - 1152000
 * - 1500000
 * - 2000000
 * - 2500000
 * - 3000000
 * - 3500000
 * - 4000000
 *
 * @ingroup CommunicationModule
 */
class SerialPort
{
public:
	typedef boost::asio::serial_port::parity Parity;
	typedef boost::asio::serial_port::stop_bits StopBits;
	typedef boost::asio::serial_port::flow_control FlowControl;
	typedef boost::asio::serial_port::character_size CharacterSize;

public:
	/**
	 * Construct a new serial port with the given I/O service.
	 * @param [in] service The I/O service object.
	 * @param [in] device The device name of the serial port.
	 * @param [in] baudrate The requested baudrate.
	 */
#if BOOST_VERSION < 106600
	SerialPort(boost::asio::io_service& service, const std::string& device,
#else
	SerialPort(boost::asio::io_context& service, const std::string& device,
#endif
	           uint32 baudrate) :
		mSerialPort(service, device)
	{
		setBaudrate(baudrate);
	}

	/// The destructor.
	virtual ~SerialPort()
	{
		mSerialPort.close();
	}

	/// conversion to boost::asio::serial_port
	operator boost::asio::serial_port&()
	{
		return mSerialPort;
	}

	/// conversion to boost::asio::serial_port
	operator const boost::asio::serial_port&() const
	{
		return mSerialPort;
	}

	/// Access to the underlying boost::asio::serial_port object.
	boost::asio::serial_port& port()
	{
		return mSerialPort;
	}

	/// Access to the underlying boost::asio::serial_port object.
	const boost::asio::serial_port& port() const
	{
		return mSerialPort;
	}

	/// Set a new baudrate.
	SerialPort& setBaudrate(uint32 rate)
	{
#if BOOST_VERSION < 104700
		Baudrate baudrate(mSerialPort.native(), rate);
#else
		Baudrate baudrate(mSerialPort.native_handle(), rate);
#endif
		mSerialPort.set_option(baudrate);
		return *this;
	}

	/// Set a new parity.
	SerialPort& setParity(Parity::type parity)
	{
		mSerialPort.set_option(Parity(parity));
		return *this;
	}

	/// Set a new stop bits configuration.
	SerialPort& setStopBits(StopBits::type stopbits)
	{
		mSerialPort.set_option(StopBits(stopbits));
		return *this;
	}

	/// Set a new data bits configuration.
	SerialPort& setDataBits(uint32 databits)
	{
		mSerialPort.set_option(CharacterSize(databits));
		return *this;
	}

	/// Set a new flow control (handshake) configuration.
	SerialPort& setFlowControl(FlowControl::type flowcontrol)
	{
		mSerialPort.set_option(FlowControl(flowcontrol));
		return *this;
	}

protected:
	/// The underlying boost::asio::serial_port object.
	boost::asio::serial_port mSerialPort;

protected:
	/// An internal class for handling the baudrate.
	class Baudrate
	{
	public:
#if BOOST_VERSION < 104700
		explicit Baudrate(boost::asio::serial_port::native_type handle,
#else
		explicit Baudrate(boost::asio::serial_port::native_handle_type handle,
#endif
		                  uint32 rate = 0) :
			mHandle(handle), mRate(rate)
		{}
		uint32 rate() const
		{
			return mRate;
		}

		boost::system::error_code store(MIRA_ASIO_OPTION_STORAGE& storage,
			boost::system::error_code& ec) const;
		boost::system::error_code load(const MIRA_ASIO_OPTION_STORAGE& storage,
			boost::system::error_code& ec)
		{
			ec = boost::system::error_code();
			return ec;
		}
	private:
#if BOOST_VERSION < 104700
		boost::asio::serial_port::native_type mHandle;
#else
		boost::asio::serial_port::native_handle_type mHandle;
#endif
		uint32 mRate;
	};
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
