/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MandelbrotSet.C
 *    Example that displays the Mandelbrot set using the Jet-Color palette.
 *
 * @author Erik Einhorn
 */


#include <geometry/Point.h>
#include <geometry/Size.h>
#include <image/Img.h>
#include <image/Colormap.h>
#include <opencv2/highgui/highgui.hpp>

using namespace std;
using namespace mira;

///////////////////////////////////////////////////////////////////////////////

int main()
{
	// set image size, scale ...
	const Size2i size(640,480);
	const Point2i origin(430, 240);
	const float scale = 0.005f;
	// ... and max. number of iterations for computing the Mandelbrot set
	const int maxIterations = 128;
	const float escapeR = 4.0f;

	// create an empty color image
	Img8U3 img(size);

	// we use the a jet color for coloring the pixels outside of the Mandelbrot set
	JetColormap colormap;

	// iterate over the whole image (in major row order)
	for(Img8U3::iterator pixel=img.begin(); pixel!=img.end(); ++pixel)
	{
		Point2i p;
		// pixel.pos() contains the position of the iterator in the image,
		// convert it into a Point2i and compute relative position to the origin
		p = pixel.pos() - origin;

		// convert it into a complex number ...
		complex<float> c((float)p.x(), (float)p.y());

		// ... and scale it
		c*=scale;

		// now check if c is part of the Mandelbrot set by examining if
		// polynomial z = z^2 + c remains bounded (do a max of N iterations)
		complex<float> z;
		for(int n=0; n<maxIterations; ++n)
		{
			z = z*z + c;

			// if |z| goes beyond a defined "escape radius" we are diverging
			// hence c is not part of the Mandelbrot set:
			if(abs(z)>escapeR) {
				// compute Normalized Iteration Count to get some
				// smooth coloring
				float mu = (float)n - (float)(log(log(abs(z)))/M_LN2);

				// colormap.getf() takes an argument between 0 and 1 and returns
				// the corresponding color within the continuous color map.
				// Set the current pixel to that color:
				*pixel = colormap.getf(mu/28);
				break;
			}
		}
	}

	// show the result
	cv::imshow("Mandelbrot Set", (cv::Mat&)img);
	cv::waitKey();

	return 0;
}
