/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeEditor.h
 *    Tape editor view.
 *
 * @author Tim Langner
 * @date   2011/12/25
 */

#ifndef _MIRA_TAPEEDITORLINE_H_
#define _MIRA_TAPEEDITORLINE_H_

#include <fw/TapeVisitor.h>

#include <TapeThumbView.h>
#include <TapeView.h>
#include <TapeChannelInfo.h>

#include <QGraphicsView>
#include <QGraphicsRectItem>
#include <QGraphicsLineItem>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

class TapeCommand;

class TapeEditor: public QWidget
{
	Q_OBJECT;
private:
	static const int sResolution[];
	static const int sTimeStep[];
	static const int sSubTimeSteps[];
public:
	TapeEditor(QWidget* parent);
	~TapeEditor();

	void clear();
	void updateVisitor();
	void setResolution(int res);
	void updateContents(bool refreshAll);

	int getResolution() const { return sResolution[mResolution]; }
	int getTimeStep() const { return sTimeStep[mResolution]; }
	int getSubTimeSteps() const { return sSubTimeSteps[mResolution]; }
	Time getRecordingTime() const { return mRecordingTime; }
	int64 getFirstMessageTime() const;
	int64 getLastMessageTime() const;
	int64 getTotalTime() const;
	int getTotalLength() const { return getTotalTime() / sResolution[mResolution]; }
	int getPixelFromTime(int64 t) const { return (t - getFirstMessageTime()) / sResolution[mResolution]; }
	int64 getTimeFromPixel(int pixel) const { return (int64)pixel * (int64)sResolution[mResolution] + getFirstMessageTime(); }

	QString getStringFromTime(int64 t, bool resolutionDependent = false);

	void setSelectionStart(int64 start);
	int64 getSelectionStart() const { return mSelectionStart; }
	int getSelectionStartX() const { return getPixelFromTime(mSelectionStart); }
	void setSelectionEnd(int64 end);
	int64 getSelectionEnd() const { return mSelectionEnd; }
	int getSelectionEndX() const { return getPixelFromTime(mSelectionEnd); }

	void clearSelectedChannels();

	TapeThumbView* getThumbView() { return mThumbView; }
	TapeDataView* getDataView() { return mTapeView->getDataView(); }
	TapeChannelView* getChannelView() { return mTapeView->getChannelView(); }
	TapeTimeView* getTimeView() { return mTapeView->getTimeView(); }
	MetaTypeDatabase& getMetaDatabase() { return mMetaDB; }

	const TapeChannelInfoMap&  getChannels() const { return mChannels; }
	TapeChannelInfoMap&  getChannels() { return mChannels; }
	void addChannel(const std::string& name,
	                const std::string& type,
	                TypeMetaPtr meta = TypeMetaPtr(),
	                const MetaTypeDatabase& db = MetaTypeDatabase() );

	void executeCommand(TapeCommand* command);

public:

	QAction* getOpenAction() { return mOpenAct; }
	QAction* getSaveAction() { return mSaveAct; }
	QAction* getEraseMessagesAction() { return mEraseMessagesAct; }
	QAction* getEraseTimeESAction()   { return mEraseTimeESAct; }
	QAction* getEraseTimeSEAction()   { return mEraseTimeSEAct; }
	QAction* getTrimGapsAction() { return mTrimGapsAct; }
	QAction* getUndoAction() { return mUndoAct; }
	QAction* getRedoAction() { return mRedoAct; }

public slots:

	void openFiles(const QStringList& files);
	void open();
	void save();
	void zoomIn(int numSteps = 1);
	void zoomOut(int numSteps = 1);
	void zoomReset();
	void eraseSelectedMessages();
	void eraseSelectedTimeEndToStart();
	void eraseSelectedTimeStartToEnd();
	void trimGaps();
	void undo();
	void redo();


protected:

	virtual void wheelEvent (QWheelEvent* event);
	void updateActionStates();

	std::list<Tape*> mTapes;
	QStringList mTapeFiles;
	TapeVisitor mVisitor;
	Time mRecordingTime;
	int64 mFirstMessage;
	int64 mLastMessage;
	TapeChannelInfoMap mChannels;
	TapeThumbView* mThumbView;
	TapeView* mTapeView;
	int mResolution;
	int64 mSelectionStart;
	int64 mSelectionEnd;
	Time mTapeStartTime;
	MetaTypeDatabase mMetaDB;

	std::deque<TapeCommand*> mUndoList;
	std::deque<TapeCommand*> mRedoList;

	QAction* mOpenAct;
	QAction* mSaveAct;
	QAction* mEraseMessagesAct;
	QAction* mEraseTimeESAct;
	QAction* mEraseTimeSEAct;
	QAction* mTrimGapsAct;
	QAction* mUndoAct;
	QAction* mRedoAct;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
