/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AddRepositoryDialog.h
 *    Implementation of AddRepositoryDialog.
 *
 * @author Christian Martin
 * @date   2012/03/26
 */

#ifndef _MIRA_ADDREPOSITORYDIALOG_H_
#define _MIRA_ADDREPOSITORYDIALOG_H_

#include <ui_AddRepositoryDialog.h>

#include <QDialog>
#include <QFileDialog>
#include <QMessageBox>
#include <QRadioButton>
#include <QStyle>

#include <core/MIRAPackage.h>
#include <core/Repository.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class AddRepositoryDialog :
	public QDialog,
	protected Ui::AddRepositoryDialog
{
	Q_OBJECT

public:
	enum RepoInfoSourceType {
		MANUALLY = 0,
		FILE,
		URL
	};

public:
	AddRepositoryDialog( QWidget* parent ) :
		QDialog( parent )
	{
		Ui::AddRepositoryDialog::setupUi(this);

		QStyle* tStyle = QApplication::style();
		mBtFileOpen->setIcon(tStyle->standardIcon(QStyle::SP_FileDialogStart));

		connect(mRbReadFromURL, SIGNAL(clicked(bool)),
		        this, SLOT(onReadFromURLClicked(bool)));
		connect(mRbReadFromFile, SIGNAL(clicked(bool)),
		        this, SLOT(onReadFromFileClicked(bool)));
		connect(mRbManually, SIGNAL(clicked(bool)),
		        this, SLOT(onManuallyClicked(bool)));
		connect(mBtFileOpen, SIGNAL(pressed()), this, SLOT(fileOpenClicked()));
		onReadFromURLClicked(true);
	}

	~AddRepositoryDialog()
	{
	}

	////////////////////////////////////////////////////////////////////////////

public slots:

	virtual void accept()
	{
		if (mRbReadFromURL->isChecked()) {
			QUrl url(mLeURL->text());
			if (!url.isValid()) {
				QMessageBox::critical(this, "Repository error",
				             QString("The URL \"%1\" is not valid.").
				             arg(mLeURL->text()));
				return;
			}
			mInfoSourceType = URL;
			mInfoSource = mLeURL->text().trimmed();
		} else
		if (mRbReadFromFile->isChecked()) {
			QString fileName = mLeFileName->text();
			if (!QFile::exists(fileName)) {
				QMessageBox::critical(this, "Repository error",
				             QString("The file \"%1\" does not exist.").
				                     arg(fileName));
				return;
			}
			mInfoSourceType = FILE;
			mInfoSource = mLeFileName->text().trimmed();
		} else
		if (mRbManually->isChecked()) {
			mInfoSourceType = MANUALLY;
			mInfoSource = "";
		}

		QDialog::accept();
	}

public:
	RepoInfoSourceType getInfoSourceType() const
	{
		return(mInfoSourceType);
	}

	const QString& getInfoSource() const
	{
		return(mInfoSource);
	}

protected slots:

	void onReadFromURLClicked(bool checked)
	{
		mLeURL->setEnabled(true);
		mLeFileName->setEnabled(false);
		mBtFileOpen->setEnabled(false);
	}

	void onReadFromFileClicked(bool checked)
	{
		mLeURL->setEnabled(false);
		mLeFileName->setEnabled(true);
		mBtFileOpen->setEnabled(true);
	}

	void onManuallyClicked(bool checked)
	{
		mLeURL->setEnabled(false);
		mLeFileName->setEnabled(false);
		mBtFileOpen->setEnabled(false);
	}

	void fileOpenClicked()
	{
		QString fileName =
				QFileDialog::getOpenFileName(this, tr("Open file"), ".",
				                             tr("Repository file (*.repo)"));
		if (fileName.isNull())
			return;
		mLeFileName->setText(fileName);
	}

private:
	RepoInfoSourceType mInfoSourceType;
	QString mInfoSource;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
