/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SVNRepository.C
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2011/08/30
 */
#include "core/SVNRepository.h"

#include <QDir>

#include "svn_pools.h"
#include "svn_utf.h"
#include "svn_cmdline.h"
#include "svn_io.h"

#include <boost/regex.hpp>

#include "core/Tools.h"
#include "core/Package.h"
#include "core/PackageParser.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

string svnErrorMessage( svn_error_t* err, string const& prefix = string() )
{
	string message = prefix;
	if ( err ) {
		message += string(message.size() ? "\n" : "") + err->message;
		message = svnErrorMessage( err->child, message );
	}
	return message;
}

SVNRepository::SVNRepository() : Repository(), pool(NULL)
{
	initializeSVN();

	name = getClass().getMetaInfo("RepoType");
	url += name;
}

SVNRepository::~SVNRepository()
{
	svn_pool_destroy(pool);
}

void SVNRepository::setUrl( Url const& iUrl )
{
	Url tUrl = url;
	try {
		Repository::setUrl( iUrl );
	}
	catch( Exception &ex ) {
		url = tUrl;
		MIRA_RETHROW( ex, "");
	}

	std::string protocol = url.substr( 0, url.find(":") );

	if ( protocol != "http" && protocol != "https" ) {
		MIRA_THROW( XLogical, "Dont know how to handle protocol " << protocol );
	}
}

void SVNRepository::deepExamine( vector<Package*>& oPackages,
		boost::function<void (std::string const&,int,int)> progressListener )
{
	list<Url> list;
	list = svnLs( url );

	// count the interesting files
	int totalfiles = 0;
	if ( progressListener ) {
		foreach(Url const& u, list) {
			if ( u.empty() )
				continue;
			Path p( u );
			if(p.extension()==".package" || p.extension()==".extpackage"
					|| p.filename()=="mountdir.xml") {
				++totalfiles;
			}
		}
	}

	int progress = 0;
	// look for *.package files and for mountdir.xml files
	foreach(Url const& u, list)
	{
		if ( u.empty() )
			continue;
		Path p( u );
		if(p.extension()==".package" || p.extension()==".extpackage"
				|| p.filename()=="mountdir.xml") {
			if (p.filename()=="mountdir.xml") {
				string fileContent = catFile( u );
				parseMountDirXml( url / u, fileContent );
			}
			else {
				Package* tPackage = indexWebPackage(u, this);
				if ( tPackage )
					oPackages.push_back( tPackage );
			}
			progress++;
			if ( progressListener )
				progressListener(u, progress, totalfiles );
		}
	}
}

string SVNRepository::catFile( Url const& relUrl )
{
	svn_error_t* err;
	svn_opt_revision_t revision;
	svn_stream_t *out;
	svn_stringbuf_t *sbuf;

	revision.kind = svn_opt_revision_head;

	string tUrl = url / relUrl;

	const char* normalPath = tUrl.c_str();
	const char* utf8path;

	apr_pool_t* subpool = svn_pool_create(pool);
	apr_pool_t* scratchpool = svn_pool_create(pool);

	svn_utf_cstring_to_utf8(&utf8path, normalPath, subpool);

	sbuf = svn_stringbuf_create("", subpool);
	out = svn_stream_from_stringbuf(sbuf, subpool);

	/* Now do the real work. */

#if SVN_VER_MINOR >= 9
	err = svn_client_cat3(NULL, // props
	                      out,
	                      utf8path,
	                      &revision,
	                      &revision,
	                      TRUE, // expand_keywords
	                      ctx,
	                      subpool,
	                      scratchpool);
#else
	err = svn_client_cat2(out, utf8path, &revision, &revision, ctx, subpool);
#endif

	string tReturn;

	if ( err ) {
		prompt->showMessage( "Error", svnErrorMessage(err) );
		svn_error_clear(err);
	}
	else
		tReturn = sbuf->data;

	apr_pool_destroy(scratchpool);
	apr_pool_destroy(subpool);

	return tReturn;
}
	
bool SVNRepository::isResponsibleFor( Path const& localPath )
{
	svn_error_t* err;

	string localPathStr = localPath.generic_string();
	const char* normalPath = localPathStr.c_str();
	const char* utf8path;

	apr_pool_t* subpool = svn_pool_create(pool);

	svn_utf_cstring_to_utf8(&utf8path, normalPath, subpool);

	/* Now do the real work. */

#if SVN_VER_MINOR >= 9
	err = svn_client_info4( utf8path, // abspath_or_url
	                  NULL, //peg_revision,
	                  NULL, //revision,
	                  svn_depth_empty, // depth,
	                  FALSE, // fetch_excluded,
	                  FALSE, // fetch_actual_only,
	                  FALSE, // include_externals,
	                  NULL, // changelists,
	                  receiveSvnInfo2, // receiver,
	                  this, // receiver_baton,
	                  ctx, //ctx,
	                  subpool //scratch_pool
	                  );
#elif SVN_VER_MINOR >= 7
	err = svn_client_info3( utf8path, // abspath_or_url
	                  NULL, //peg_revision,
	                  NULL, //revision,
	                  svn_depth_empty, // depth,
	                  FALSE, // fetch_excluded,
	                  FALSE, // fetch_actual_only,
	                  NULL, // changelists,
	                  receiveSvnInfo2, // receiver,
	                  this, // receiver_baton,
	                  ctx, //ctx,
	                  subpool //scratch_pool
	                  );
#else
	err = svn_client_info2(utf8path, NULL, NULL, receiveSvnInfo,
			this, svn_depth_empty, NULL, ctx, subpool);
#endif


	if (err) {
		// ignore error here
		svn_error_clear(err);
		apr_pool_destroy(subpool);
		return false;
	}

	if ( isGeneric() )
		return true;

	apr_pool_destroy(subpool);

	return ( mInfoURL.size() > url.size()
			&& mInfoURL.substr( 0, url.size()+1 ) == url+"/" );
}

Path SVNRepository::getRelativePathFor( Path const& localPath )
{
	if ( !isResponsibleFor( localPath ) ) {
		return "";
	}

	if ( isGeneric() )
		return mInfoURL;

	return mInfoURL.substr( url.size()+1 );
}

Url SVNRepository::getMountDirForPackage( Url const& subUrl, std::string* matchedUrl )
{
	string tMatchedUrl;
	Url packageSubPath = Repository::getMountDirForPackage( subUrl, &tMatchedUrl );

	if ( matchedUrl )
		*matchedUrl = tMatchedUrl;

	// we have to append the sub path of the svn repository...
	if ( packageSubPath.size() ) {

		if ( (url / subUrl).substr( tMatchedUrl.size(), string::npos ).find('/') != string::npos ) {
			// we can find at least one subdir -> append the last one
			//packageSubPath += subUrl.substr( subUrl.rfind('/'), string::npos );
			// changed: add the full subdir
			packageSubPath += (url / subUrl).substr( tMatchedUrl.size(), string::npos );
		}
	}

	return packageSubPath;

}

void SVNRepository::install( Package const& package, Path const& destPath )
{
	Url tSourceUrl = getParentUrl( url / package.mRepoSubPath / package.mFileName );

	svn_error_t* err;
	svn_opt_revision_t revision;
	svn_revnum_t* resultRev = NULL;

	// create the new directory and CMakeLists.txt if necessary
	prepareCheckoutDirectory( destPath, package.mType & Package::SOURCE );

	string destPathStr = destPath.string();
	const char* NormalUrl = tSourceUrl.c_str();
	const char* URL;
	const char* NormalPath = destPathStr.c_str();
	const char* PATH;

	apr_pool_t* subpool = svn_pool_create(pool);

	svn_utf_cstring_to_utf8(&URL, NormalUrl, subpool);
	svn_utf_cstring_to_utf8(&PATH, NormalPath, subpool);

	/* Now do the real work. */

	/* Set revision to always be the HEAD revision.  It could, however,
	 * be set to a specific revision number, date, or other values. */
	revision.kind = svn_opt_revision_head;

	/* Main call into libsvn_client does all the work. */
	err = svn_client_checkout3( resultRev, URL, PATH, &revision, &revision,
		svn_depth_infinity, false, true, ctx, subpool );

	if (err) {
		prompt->showMessage( "Error", svnErrorMessage( err ) );
		svn_error_clear(err);
	}

	apr_pool_destroy(subpool);
}

string statusToString( svn_wc_status_kind const& status )
{
	switch ( status ) {
	case svn_wc_status_unversioned: return "unversioned";break;
	case svn_wc_status_added: return "added";break;
	case svn_wc_status_missing: return "missing"; break;
	case svn_wc_status_deleted: return "deleted";break;
	case svn_wc_status_replaced: return "replaced";break;
	case svn_wc_status_modified: return "modified";break;
	case svn_wc_status_merged: return "merged";break;
	case svn_wc_status_conflicted: return "conflicting";break;
	default: return "unknown";
	}
	return "unknown";
}

void removeDir(QDir const& dir)
{
	if ( dir.exists() ) {
		QFileInfoList entries = dir.entryInfoList(QDir::NoDotAndDotDot | QDir::Dirs | QDir::Files | QDir::Hidden);
		for (int i = 0; i < entries.size(); i++) {
			QFileInfo entry = entries[i];
			QString path = entry.absoluteFilePath();
			if (entry.isDir())
				removeDir( QDir(path) );
			else {
				QFile file(path);
				if (!file.remove())
					MIRA_THROW(XIO,"Error removing file "+path.toStdString()+"!")
			}
		}
		if (!dir.rmdir(dir.absolutePath()))
			MIRA_THROW(XIO,"Error removing directory "+dir.absolutePath().toStdString());
	}
}

void SVNRepository::uninstall( Package const& package )
{
	Path localPath = package.mLocalPath / package.mFileName;

	MIRA_LOG(NOTICE) << "uninstall " << localPath.string();
	Path tPackagePath = localPath;
	tPackagePath.remove_filename();
	MIRA_LOG(NOTICE) << "  path: " << tPackagePath;

	if (svnStat( tPackagePath ) ) {
		// check if uninstall is save
		if ( mStatusInfo.size() > 0 ) {
			string mRootURL = mStatusInfo[0].URL;
			string errorMsg;
			foreach( SVNStatusInfo const& info, mStatusInfo ) {
				// assure that the file path starts with mRootURL
				// and that all the file have the status normal (not modified or added or ...).
				// however, we should ignore core files and backup files created
				// by text editors

				// match core + optional .(Number)* zero or one time
				boost::regex corefile("core(.[0-9]*)?+");
				// ignore ignored resources, files with tailing ~ and core files
				if ( info.status != svn_wc_status_ignored
						&& (info.name.length() > 0 && info.name[ info.name.length()-1 ] != '~' )
						&& ( !boost::regex_match(Path(info.name).filename().string(), corefile ) )
						// assure that the resource has state normal and that the url is contained
						// within the parent url
						&& ( info.URL.find( mRootURL ) != 0 || info.status != svn_wc_status_normal )  ) {
					errorMsg += statusToString(info.status) + " " + info.name + "\n";
				}
			}
			if ( errorMsg.empty() ) {
				// remove libs and manifests...
				// go into associated mira lib directory and remove all files
				// linking to the directory to be removed
				removeGeneratedLibraryFiles( package );

				// remove the whole directory
				QDir tDir( QString::fromStdString( tPackagePath.string() ) );
				MIRA_LOG(NOTICE) << "remove dir " << tDir.absolutePath().toStdString();
				removeDir( tDir );
				MIRA_LOG(NOTICE) << "do uninstall!";
			}
			else {
				prompt->showMessage( "Error", "Uninstall is blocked by the following conflicts:\n"+errorMsg );
			}
		}
	}
	// reinitialize svn context, since it keeps track of the deleted folder and will deny new checkout to this location
	initializeSVN();
}

void SVNRepository::initializeSVN()
{
	if ( pool )
		svn_pool_destroy(pool);

	svn_error_t* err;

	// Create top-level memory pool.
	pool = svn_pool_create (NULL);

	// Make sure the ~/.subversion run-time config files exist
	err = svn_config_ensure (NULL, pool);
	if (err) {
		MIRA_LOG(ERROR) << svnErrorMessage(err);
	}

	// Initialize and allocate the client_ctx object.
#if SVN_VER_MINOR >= 8
	if ((err = svn_client_create_context2(&ctx, NULL, pool))) {
#else
	if ((err = svn_client_create_context (&ctx, pool))) {
#endif
		MIRA_LOG(ERROR) << svnErrorMessage(err);
	}

	// Load the run-time config file into a hash
	if ((err = svn_config_get_config (&(ctx->config), NULL, pool))) {
		MIRA_LOG(ERROR) << svnErrorMessage(err);
	}

	/* Make the client_ctx capable of authenticating users */
	{
	  /* There are many different kinds of authentication back-end
		 "providers".  See svn_auth.h for a full overview.

		 If you want to get the auth behavior of the 'svn' program,
		 you can use svn_cmdline_setup_auth_baton, which will give
		 you the exact set of auth providers it uses.  This program
		 doesn't use it because it's only appropriate for a command
		 line program, and this is supposed to be a general purpose
		 example. */

	  svn_auth_provider_object_t *provider;
	  apr_array_header_t *providers
		= apr_array_make (pool, 2, sizeof (svn_auth_provider_object_t *));

	  svn_auth_get_simple_prompt_provider (&provider,
			  SVNRepository::my_simple_prompt_callback,
			  this, /* baton */ 2, /* retry limit */ pool);
	  APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;

	  svn_auth_get_ssl_server_trust_prompt_provider (&provider,
			  SVNRepository::my_simple_sslserver_prompt_callback,
			  this, pool);
	  APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;

	  /* Register the auth-providers into the context's auth_baton. */
	  svn_auth_open (&ctx->auth_baton, providers, pool);
	}
}

bool getPassword( svn_auth_cred_simple_t *cred,
        apr_hash_t* creds_hash )
{
	svn_string_t *str;
	str = static_cast<svn_string_t*>(apr_hash_get (creds_hash,
			"password", APR_HASH_KEY_STRING));
	if ( str && str->data ) {
		cred->password = str->data;
		return true;
	}
	return false;
}

/* A tiny callback function of type 'svn_auth_simple_prompt_func_t'. For
   a much better example, see svn_cl__auth_simple_prompt in the official
   svn cmdline client. */
svn_error_t *
SVNRepository::my_simple_prompt_callback (svn_auth_cred_simple_t **cred,
                           void *baton,
                           const char *realm,
                           const char *username,
                           svn_boolean_t may_save,
                           apr_pool_t *pool)
{
	Repository* tThis = (Repository*)baton;
	assert( tThis );

	svn_auth_cred_simple_t *ret =
			(svn_auth_cred_simple_t*)apr_pcalloc(pool, sizeof (*ret));

	// Try to load existing credential from svn config first of all
	svn_error_t *err;
	apr_hash_t *creds_hash = NULL;

	// try to fill credential hash for given realm
	err = svn_config_read_auth_data (&creds_hash, SVN_AUTH_CRED_SIMPLE,
		  	  	  realm, NULL, pool);

	svn_error_clear (err);

	if (! err && creds_hash) {
		// extract username and password from the credential hash
		svn_string_t *str;
		str = static_cast<svn_string_t*>(apr_hash_get (creds_hash,
				"username", APR_HASH_KEY_STRING));
		if ( str && str->data )
			ret->username = str->data;

		str = static_cast<svn_string_t*>(apr_hash_get(creds_hash,
				"passtype", APR_HASH_KEY_STRING));
		if (str && str->data) {
			getPassword( ret, creds_hash );

			if ( ret->username && ret->password ) {
				*cred = ret;
				return SVN_NO_ERROR;
			}
		}
	}
	// ask the user to provide username and password
	if ( tThis->prompt ) {
		tThis->credential.realm = realm ? realm : string();
		tThis->credential.user = username ? username : string();
		if ( tThis->prompt->getLogin( tThis->credential ) ) {
			ret->username = tThis->credential.user.c_str();
			ret->password = tThis->credential.password.c_str();
			*cred = ret;
		}
	}

	return SVN_NO_ERROR;
}

/*
 * source: http://tortoisesvn.googlecode.com/svn/trunk/src/SVN/SVNPrompt.cpp
 */
svn_error_t* SVNRepository::my_simple_sslserver_prompt_callback(svn_auth_cred_ssl_server_trust_t **cred_p,
							void *baton,
							const char *realm,
							apr_uint32_t failures,
							const svn_auth_ssl_server_cert_info_t *cert_info,
							svn_boolean_t may_save,
							apr_pool_t *pool)
{
	Repository* tThis = (Repository*)baton;
	bool trusted = false;

	apr_hash_t* creds_hash = NULL;

	// Check if this is a permanently accepted certificate
	svn_error_t* err =
			svn_config_read_auth_data (&creds_hash, SVN_AUTH_CRED_SSL_SERVER_TRUST,
			realm, NULL, pool);
	svn_error_clear (err);
	if ( !err && creds_hash ) {
		svn_string_t *trusted_cert, *this_cert, *failstr;
		apr_uint32_t last_failures = 0;

		trusted_cert = static_cast<svn_string_t*>(
				apr_hash_get (creds_hash, "ascii_cert", APR_HASH_KEY_STRING));
		this_cert = svn_string_create (cert_info->ascii_cert, pool);
		failstr = static_cast<svn_string_t*>(
				apr_hash_get (creds_hash, "failures", APR_HASH_KEY_STRING));

		if (failstr) {
			char *endptr;
			unsigned long tmp_ulong = strtoul (failstr->data, &endptr, 10);

			if (*endptr == '\0')
				last_failures = (apr_uint32_t) tmp_ulong;
		}

		/* If the cert is trusted and there are no new failures, we
		* accept it by clearing all failures. */
		if (trusted_cert && svn_string_compare (this_cert, trusted_cert) &&
				(failures & ~last_failures) == 0) {
			trusted = true;
		}
	}

	// If all failures are cleared now, we return the creds
	if (!failures || trusted)
	{
		*cred_p = (svn_auth_cred_ssl_server_trust_t*)apr_pcalloc (pool, sizeof(**cred_p));
		(*cred_p)->may_save = FALSE; /* No need to save it again... */
		(*cred_p)->accepted_failures = failures;
	}
	else if ( tThis->prompt ) {
		if (failures & SVN_AUTH_SSL_UNKNOWNCA) {

			switch( tThis->prompt->acceptServerCertificate(*cert_info ) )
			{
			case PromptProvider::Reject : break;
			case PromptProvider::Temporary :
				*cred_p = (svn_auth_cred_ssl_server_trust_t*)apr_pcalloc (pool, sizeof (**cred_p));
				(*cred_p)->may_save = FALSE;
				(*cred_p)->accepted_failures = failures;
				break;
			case PromptProvider::Always :
				*cred_p = (svn_auth_cred_ssl_server_trust_t*)apr_pcalloc (pool, sizeof (**cred_p));
				(*cred_p)->may_save = TRUE;
				(*cred_p)->accepted_failures = failures;
				break;
			}
		}
	}
	return SVN_NO_ERROR;
}

svn_error_t* SVNRepository::receiveSvnInfo(void* baton, const char* path, const svn_info_t* info, apr_pool_t*)
{
	SVNRepository* tThis = (SVNRepository*)baton;
	tThis->mInfoRoot = info->repos_root_URL;
	tThis->mInfoURL = info->URL;
	return NULL;
}

#if SVN_VER_MINOR >= 7
svn_error_t* SVNRepository::receiveSvnInfo2(void* baton, const char* path, const svn_client_info2_t* info, apr_pool_t*)
{
	SVNRepository* tThis = (SVNRepository*)baton;
	tThis->mInfoRoot = info->repos_root_URL;
	tThis->mInfoURL = info->URL;
	return NULL;
}
#endif

svn_error_t* SVNRepository::receiveSvnStatus(void* baton, const char* path, svn_wc_status2_t* status, apr_pool_t*)
{
	SVNRepository* tThis = (SVNRepository*)baton;
	tThis->mStatusInfo.push_back( SVNStatusInfo( path, status->entry ? status->entry->url : "", status->text_status ) );
	return NULL;
}

#if SVN_VER_MINOR >= 7
svn_error_t* SVNRepository::receiveClientSvnStatus( void *baton,
                                                    const char *path,
                                                    const svn_client_status_t *status,
                                                    apr_pool_t *scratch_pool )
{
	SVNRepository* tThis = (SVNRepository*)baton;
	std::string relpath;
	if (status->repos_relpath != NULL)
		relpath = std::string(status->repos_relpath);
	tThis->mStatusInfo.push_back( SVNStatusInfo( path, relpath, status->text_status ) );
	return NULL;
}
#endif

list<Url> SVNRepository::svnLs( Url const& url )
{
	list<Url> tReturn;

	svn_error_t *err;
	svn_opt_revision_t revision;

	if(url.empty())
		return tReturn;

	const char *NormalUrl = url.c_str();
	const char *URL;

	apr_pool_t* subpool = svn_pool_create(pool);

	svn_utf_cstring_to_utf8(&URL, NormalUrl, subpool);

	/* Now do the real work. */

	/* Set revision to always be the HEAD revision.  It could, however,
	 * be set to a specific revision number, date, or other values. */
	revision.kind = svn_opt_revision_head;

	/* Main call into libsvn_client does all the work. */
#if SVN_VER_MINOR >= 10
	err = svn_client_list4(URL, // path_or_url,
	                 &revision, // peg_revision,
	                 &revision, // revision,
	                 NULL,      // patterns
	                 svn_depth_infinity, // depth,
	                 0, // dirent_fields,
	                 FALSE, // fetch_locks,
	                 FALSE, // include_externals,
	                 receiveListInfo2, // list_func,
	                 &tReturn, // baton,
	                 ctx, // ctx,
	                 subpool //apr_pool_t *pool
	                 );
#elif SVN_VER_MINOR >= 8
	err = svn_client_list3(URL, // path_or_url,
	                 &revision, // peg_revision,
	                 &revision, // revision,
	                 svn_depth_infinity, // depth,
	                 0, // dirent_fields,
	                 FALSE, // fetch_locks,
	                 FALSE, // include_externals,
	                 receiveListInfo2, // list_func,
	                 &tReturn, // baton,
	                 ctx, // ctx,
	                 subpool //apr_pool_t *pool
	                 );
#else
	err = svn_client_list2(URL, &revision, &revision, svn_depth_infinity,
			/*SVN_DIRENT_ALL*/0, FALSE, receiveListInfo, &tReturn, ctx, subpool);
#endif

	if (err) {
		prompt->showMessage( "Error", svnErrorMessage(err) );
	}

	svn_error_clear(err);
	apr_pool_destroy(subpool);

	return tReturn;
}

bool SVNRepository::svnStat( mira::Path const& path )
{
	mStatusInfo.clear();

	svn_error_t* err;
	svn_opt_revision_t revision;
	svn_revnum_t* resultRev = NULL;

	string pathStr = path.string();
	const char* PATH;

	apr_pool_t* subpool = svn_pool_create(pool);

	svn_utf_cstring_to_utf8(&PATH, pathStr.c_str(), subpool);

	/* Now do the real work. */

	/* Set revision to always be the HEAD revision.  It could, however,
	 * be set to a specific revision number, date, or other values. */
	revision.kind = svn_opt_revision_head;

#if SVN_VER_MINOR >= 9
	err = svn_client_status6( resultRev, //svn_revnum_t *result_rev,
						ctx, // ctx,
						PATH, // path,
						&revision, // revision,
						svn_depth_infinity, // depth,
						TRUE, // get_all,
						FALSE, // check_out_of_date,
						TRUE, // check_working_copy (ignored if check_out_of_date==FALSE),
						FALSE, // no_ignore,
						FALSE, // ignore_externals,
						FALSE, // depth_as_sticky,
						NULL, // changelists,
						receiveClientSvnStatus, // status_func,
						this, // status_baton,
						subpool // scratch_pool
						);
#elif SVN_VER_MINOR >= 7
	err = svn_client_status5( resultRev, //svn_revnum_t *result_rev,
						ctx, // ctx,
						PATH, // path,
						&revision, // revision,
						svn_depth_infinity, // depth,
						TRUE, // get_all,
						FALSE, // update,
						FALSE, // no_ignore,
						FALSE, // ignore_externals,
						FALSE, // depth_as_sticky,
						NULL, // changelists,
						receiveClientSvnStatus, // status_func,
						this, // status_baton,
						subpool // scratch_pool
						);
#else
	err = svn_client_status4( resultRev, PATH, &revision,
						receiveSvnStatus /* status_func */,
						this /* status_baton */,
						svn_depth_infinity /* depth */,
						TRUE /*get_all*/,
						FALSE /*update*/,
						FALSE /*no_ignore*/,
						FALSE /*ignore_externals*/,
						NULL /* changelists*/,
						ctx,
						subpool);
#endif

	if (err) {
		prompt->showMessage( "Error", svnErrorMessage(err) );
		svn_error_clear(err);
		apr_pool_destroy(subpool);
		return false;
	}

	apr_pool_destroy(subpool);
	return true;
}

svn_error_t* SVNRepository::receiveListInfo(void* baton, const char* path, const svn_dirent_t* /*dirent*/, const svn_lock_t* /*lock*/, const char* /*abs_path*/, apr_pool_t* /*pool*/)
{
	list<Url>& files = *reinterpret_cast<list<Url>*>(baton);
	files.push_back(path);
	return NULL;
}

svn_error_t* SVNRepository::receiveListInfo2(
									 void *baton,
									 const char *path,
									 const svn_dirent_t *dirent,
									 const svn_lock_t *lock,
									 const char *abs_path,
									 const char *external_parent_url,
									 const char *external_target,
									 apr_pool_t *scratch_pool )
{
	list<Url>& files = *reinterpret_cast<list<Url>*>(baton);
	files.push_back(path);
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::SVNRepository, mira::Repository);
