/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CmdLinePromptProvider.C
 *    $short description$.
 *
 * @author Ronny Stricker
 * @date   2012/02/28
 */

#include <iostream>

#include <platform/Console.h>
#include <utils/ToString.h>
#include <utils/MakeString.h>

#include "app/CmdLinePromptProvider.h"

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

string CmdLinePromptProvider::getInput( string const& message )
{
	string output;
	cout << message;
	cout.flush();
	if (noninteractive) {
		cout << endl;
		return "";
	}
	getline( cin, output );

	if(cin.eof()){
		printf("User pressed Ctrl+D\n");
		exit(1);
	}

	return output;
}

unsigned int CmdLinePromptProvider::getSelection( string const& message, unsigned int maxSelection, unsigned int defaultSelection /* = 0 */ )
{
	do {
		string selection = getInput(MakeString() << message << " (0-" << maxSelection << ", default: " << defaultSelection << "): ");
		cout << endl;
		if (selection.empty())
			return defaultSelection;
		try {
			unsigned int selected = fromString<unsigned int>(selection);
			if (selected <= maxSelection)
				return selected;
		}
		catch (XIO&) {}
	} while (true);
}

bool CmdLinePromptProvider::getLogin( Credential& credential, bool forceUserInput /* = false */ )
{
	cout << endl;
	cout << "login for repository " << credential.realm << endl;
	if (forceUserInput || credential.user.empty() || credential.user == "anonymous")
		credential.user = getInput("user: ");
	else
		cout << "user: " << credential.user << endl;
	console::enableEcho( false );
	credential.password = getInput("password: ");
	console::enableEcho( true );
	cout << endl;
	cout << endl;
	return true;
}

PromptProvider::AcceptValue CmdLinePromptProvider::acceptServerCertificate( ServerCertificate const& certificate )
{
	cout << endl;
	cout << "Do you trust the server certificate from " << certificate.hostname << "?\n";
	cout << "Fingerprint: " << certificate.fingerprint << "\n";
	string decision = "?";
	while (!decision.empty() && decision != "y" && decision != "n" && decision != "a")
		decision = getInput("(y)es (n)o (a)lways:" );
	cout << endl;
	if ( decision == "a" )
		return Always;
	else if ( decision == "n" )
		return Reject;
	return Temporary;
}

void CmdLinePromptProvider::showMessage( string const& title, string const& message )
{
	cout << (title.empty() ? "" : "[" + title + "]: ") << message << endl;
}

bool CmdLinePromptProvider::showYesNoMessage( string const& title, string const& message, bool defaultSelection /* = true */ )
{
	cout << endl;
	showMessage( title, message );
	string decision = "?";
	while (!decision.empty() && decision != "y" && decision != "n")
		decision = getInput(MakeString() << "(y)es (n)o (default: " << (defaultSelection ? "y" : "n") << "): ");
	if (decision.empty())
		return defaultSelection;
	return decision == "y";
}

unsigned int CmdLinePromptProvider::showSelectionMessage( std::string const& title, std::string const& message,
                                                          std::vector<std::string> const& options, unsigned int defaultSelection /* = 0 */,
                                                          bool showOptions /* = true */ )
{
	cout << endl;
	showMessage( title, message );

	for (unsigned int i = 0; i < options.size(); ++i)
		cout << " " << i << ") " << options[i] << endl;

	// get user selection
	return getSelection("Selection", options.size() - 1, defaultSelection);
}

std::string CmdLinePromptProvider::showInputMessage( std::string const& title, std::string const& message, std::string const& prefix /* = "" */ )
{
	cout << endl;
	showMessage( title, message );

	// get user input
	string input;
	while (!input.length())
		input = getInput(prefix);
	return input;
}

//////////////////////////////////////////////////////////////////////////////

}
