/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformTreeTextView.C
 *    Shows structure of the transformation tree in textual form.
 *
 * @author Christof Schroeter
 * @date   2013/05/24
 */

#include <views/TransformTreeTextView.h>

#include <widgets/TreeViewFilter.h>
#include "private/TransformTreeWidget.h"

#include <widgets/QtUtils.h>

using namespace mira;

#define UPDATE_INTERVAL 2000

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TransformTreeTextView::TransformTreeTextView() : ui(NULL)
{
	mUpdateTimer = 0;
}

QWidget* TransformTreeTextView::createPartControl()
{
	QWidget* widget = new QWidget(this);
	QVBoxLayout* layout = new QVBoxLayout;

	ui = new TransformTreeWidget(widget);
	ui->setColumnCount(5);
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	ui->header()->setSectionResizeMode(QHeaderView::ResizeToContents);
#else
	ui->header()->setResizeMode(QHeaderView::ResizeToContents);
#endif
	ui->header()->setStretchLastSection(false);
	ui->header()->setDefaultAlignment(Qt::AlignHCenter);

	QStringList headerLabels;
	headerLabels << tr("Frame ID") << tr("Type") << tr("Cov")
	             << tr("Pose To Parent") << tr("Pose To Checked");
	ui->setHeaderLabels(headerLabels);

	TreeViewFilter* filter = new TreeViewFilter(ui, widget);
	filter->setExpandAll(true);
	filter->setShowChildren(false);
	filter->setFilterColumns(0);
	filter->setFocus();

	layout->addWidget(filter);
	layout->addWidget(ui);

	widget->setLayout(layout);

	updateTree();

	connect(ui, SIGNAL(itemChanged(QTreeWidgetItem*, int)),
	        ui, SLOT(itemChanged(QTreeWidgetItem*, int))   );

	// set the data memorized in reflect()
	ui->checkCov(mCheckCov);
	if (!mSelectID.empty())
		ui->selectID(mSelectID);

	mUpdateTimer = startTimer(UPDATE_INTERVAL);

	return widget;
}

void TransformTreeTextView::timerEvent(QTimerEvent *event)
{
	if (event->timerId() == mUpdateTimer)
		updateTree();
}

void TransformTreeTextView::updateTree()
{
	// find new nodes and nodes that changed their parent

	std::map<TransformerBase::AbstractNodePtr, QTreeWidgetItem*> newNodes;
	std::map<TransformerBase::AbstractNodePtr, std::string> relocateNodes;

	auto nodes = MIRA_FW.getTransformer()->getNodes();
	foreach(TransformerBase::AbstractNodePtr node, nodes) {
		// node already displayed?
		std::map<std::string, QTreeWidgetItem*>::const_iterator find = ui->items.find(node->getID());
		if (find == ui->items.end()) {
			// check if node has no parent or the parent is already displayed
			// otherwise, this node's parent is new too,
			// so that parents recursive insertion will add the current node
			// and we ignore it here
			if (node->getParent() == NULL)
				newNodes[node] = NULL;
			else {
				find = ui->items.find(node->getParent()->getID());
				if (find != ui->items.end())
					newNodes[node] = find->second;
			}
		}
		else {
			// check if parent is still the same, otherwise need to relocate
			if (find->second->parent() == NULL) {
				if (node->getParent() != NULL) // had no parent before but has now
					relocateNodes[node] = node->getParent()->getID();
			}
			else // parent changed - apparently this cannot happen anyway
				if (node->getParent() == NULL)
					relocateNodes[node] = "";
				else if (find->second->parent()->text(0).toStdString() != node->getParent()->getID())
					relocateNodes[node] = node->getParent()->getID();
		}
	}

	// add all missing first, then relocate existing ones afterwards
	// (as that might need new nodes as parents)
	foreach(auto node, newNodes)
		ui->addNode(node.first, node.second);

	foreach(auto node, relocateNodes)
		ui->relocateNode(node.first, node.second);

	// structure up-to-date now, now update all nodes' texts
	ui->updateAllLabels();
}

std::map<std::string, bool> TransformTreeTextView::getShowCovarianceMap() const
{
	if (ui)
		return ui->covChecked();

	return std::map<std::string, bool>();
}

void TransformTreeTextView::setShowCovarianceMap(const std::map<std::string, bool>& check)
{
	if (ui)
		ui->checkCov(check);
	else
		mCheckCov = check;
}

std::string TransformTreeTextView::getRelativePoseNodeID() const
{
	if (ui)
		return ui->selectedID();

	return "";
}

void TransformTreeTextView::setRelativePoseNodeID(const std::string& id)
{
	if (ui)
		ui->selectID(id);
	else
		mSelectID = id;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::TransformTreeTextView, mira::EditorPart );
