/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaTypeDatabaseView.C
 *    Implementation of MetaTypeDatabaseView.h
 *
 * @author Christof Schröter
 * @date   2019/02/17
 */

#include <views/MetaTypeDatabaseView.h>

#include <fw/Framework.h>

#include <QHeaderView>
#include <QSplitter>

#include <widgets/TreeViewFilter.h>
#include <widgets/QtUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MetaTypeDatabaseView::MetaTypeDatabaseView() : mira::EditorPart()
{
	mViewWidget = NULL;
	mTypesListView = NULL;
	mTypeInfoView = NULL;
	mUpdateTimer = 0;
}

QWidget* MetaTypeDatabaseView::createPartControl()
{
	mViewWidget = new QWidget(this);
	QHBoxLayout* layout = new QHBoxLayout(mViewWidget);

	QSplitter* splitter = new QSplitter(mViewWidget);
	splitter->setOrientation(Qt::Horizontal);
	layout->addWidget(splitter);

	QWidget* typesListWidget = new QWidget(splitter);
	QVBoxLayout* typesListLayout = new QVBoxLayout;

	mTypesListView = new QTreeWidget(typesListWidget);
	mTypesListView->setColumnCount(1);
	mTypesListView->setRootIsDecorated(false);
	mTypesListView->setAlternatingRowColors(true);
	mTypesListView->setAlternatingRowColors(true);

	QStringList headers;
	headers << tr("Types");
	mTypesListView->setHeaderLabels(headers);

	connect(mTypesListView, SIGNAL(itemSelectionChanged()),
	        this, SLOT(onSelect()));

	TreeViewFilter* typeFilter = new TreeViewFilter(mTypesListView, typesListWidget);
	typeFilter->setFilterColumns(0);
	typeFilter->setFocus();

	typesListLayout->addWidget(typeFilter);
	typesListLayout->addWidget(mTypesListView);

	typesListWidget->setLayout(typesListLayout);


	QWidget* typeInfoWidget = new QWidget(splitter);
	QVBoxLayout* typeInfoLayout = new QVBoxLayout;

	mTypeInfoView = new QTreeWidget(typeInfoWidget);
	mTypeInfoView->setColumnCount(3);
	mTypeInfoView->setRootIsDecorated(true);
	mTypeInfoView->setAlternatingRowColors(true);

	headers.clear();
	headers << "" << "" << "";
	mTypeInfoView->setHeaderLabels(headers);

	TreeViewFilter* infoFilter = new TreeViewFilter(mTypeInfoView, typeInfoWidget);
	infoFilter->setExpandAll(true);
	infoFilter->setFilterColumns(0, 1, 2);
	infoFilter->setFocus();

	typeInfoLayout->addWidget(infoFilter);
	typeInfoLayout->addWidget(mTypeInfoView);

	typeInfoWidget->setLayout(typeInfoLayout);

	splitter->addWidget(typesListWidget);
	splitter->addWidget(typeInfoWidget);
	splitter->setCollapsible(0, false);
	splitter->setCollapsible(1, false);
	splitter->setStretchFactor(1, 3);

	updateDB();

	mUpdateTimer = startTimer(1000);

	return mViewWidget;
}

void MetaTypeDatabaseView::timerEvent(QTimerEvent* event)
{
	if(event->timerId() == mUpdateTimer) {
		updateDB();
	}
}

void MetaTypeDatabaseView::updateDB()
{
	QString selectedType;
	QList<QTreeWidgetItem*> selectedItems = mTypesListView->selectedItems();
	if (selectedItems.size())
		selectedType = selectedItems.front()->text(0);

	QStringList types;
	{
		auto db = MIRA_FW.getMetaDatabase();

		for (int i = mTypesListView->topLevelItemCount(); i < (int)db->size(); ++i)
			mTypesListView->addTopLevelItem(new QTreeWidgetItem());

		// the number of meta db entries should never decrease
	//	while (mTypesListView->topLevelItemCount() > (int)db->size())
	//		delete mTypesListView->topLevelItem(0);

		foreach(const auto& v, *db)
			types << QString::fromStdString(v.first);
	}

	int row = 0;
	foreach(const QString& type, types)
	{
		QTreeWidgetItem* item = mTypesListView->topLevelItem(row);
		item->setText(0, type);
		if (type == selectedType)
			mTypesListView->setCurrentItem(item);
		++row;
	}

	mTypesListView->resizeColumnToContents(0);
}

QTreeWidgetItem* MetaTypeDatabaseView::addTypeInfoCategory(const QString& category)
{
	QTreeWidgetItem* item;

	item = new QTreeWidgetItem(mTypeInfoView, QStringList(category));
	mTypeInfoView->addTopLevelItem(item);
	QFont font = item->font(0);
	font.setBold(true);
	item->setFont(0,font);
	item->setFlags(Qt::ItemIsEnabled);
	item->setExpanded(true);
	item->setFirstColumnSpanned(true);
	mTypeInfoView->resizeColumnToContents(0);
	return item;
}

void MetaTypeDatabaseView::showMetaType(CompoundMetaPtr meta)
{
	// keep the view unchanged if meta did not change!
	static std::string prevMetaString;
	std::string metaString = meta->toString();
	if (metaString == prevMetaString)
		return;

	mTypeInfoView->clear();
	showVersions(meta);
	showMembers(meta);
	showInterfaces(meta);
	showMethods(meta);

	mTypeInfoView->resizeColumnToContents(0);
	mTypeInfoView->resizeColumnToContents(1);
	mTypeInfoView->resizeColumnToContents(2);

	prevMetaString = metaString;
}

void MetaTypeDatabaseView::showVersions(CompoundMetaPtr meta)
{
	if (meta->version.empty())
		return;

	QTreeWidgetItem* categoryItem = addTypeInfoCategory("Version");

	foreach (const auto& v, meta->version) {
		QString name = QString::fromStdString(v.first);
		if (name.isEmpty())
			name = "(anonymous)";

		QTreeWidgetItem* item
			= new QTreeWidgetItem(categoryItem, QStringList() << name << QString::number(v.second));
	}
}

void MetaTypeDatabaseView::showMembers(CompoundMetaPtr meta)
{
	if (meta->members.empty())
		return;

	QTreeWidgetItem* categoryItem = addTypeInfoCategory("Members");

	foreach(const CompoundMeta::Member& m, meta->members) {
		QTreeWidgetItem* item
			= new QTreeWidgetItem(categoryItem, QStringList() << QString::fromStdString(m.type->toString())
			                                                  << QString::fromStdString(m.name)
			                                                  << QString::fromStdString(m.comment));
	}
}

void MetaTypeDatabaseView::showInterfaces(CompoundMetaPtr meta)
{
	if (meta->interfaces.empty())
		return;

	QTreeWidgetItem* categoryItem = addTypeInfoCategory("Interfaces");

	foreach (const std::string& i, meta->interfaces) {
		QTreeWidgetItem* item
			= new QTreeWidgetItem(categoryItem, QStringList(QString::fromStdString(i)));
	}
}

void MetaTypeDatabaseView::showMethods(CompoundMetaPtr meta)
{
	if (meta->methods.empty())
		return;

	QTreeWidgetItem* categoryItem = addTypeInfoCategory("Methods");

	foreach (const MethodMetaPtr& m, meta->methods) {
		QString paramsString = "(";
		for(auto p = m->parameters.begin(); p!=m->parameters.end(); ++p)
		{
			if(p!=m->parameters.begin())
				paramsString += ",";
			paramsString += QString::fromStdString(p->type->toString());
			if (!p->name.empty())
				paramsString += " " + QString::fromStdString(p->name);
		}
		paramsString += ")";

		QTreeWidgetItem* item
			= new QTreeWidgetItem(categoryItem, QStringList() << QString::fromStdString(m->returnType->toString() + " " + m->name)
			                                                  << paramsString
			                                                  << QString::fromStdString(m->comment));

		for(auto p = m->parameters.begin(); p!=m->parameters.end(); ++p)
		{
			if (!p->name.empty() && !p->description.empty())
				QTreeWidgetItem* subitem
					= new QTreeWidgetItem(item, QStringList() << " " << QString::fromStdString(p->name)
			                                                  << QString::fromStdString(p->description));
		}
		item->setExpanded(true);
	}
}

void MetaTypeDatabaseView::onSelect()
{
	QTreeWidgetItem* selectedItem = mTypesListView->selectedItems().front();
	QString selectedType = selectedItem->text(0);

	auto db = MIRA_FW.getMetaDatabase();
	CompoundMetaPtr c = (*db)[selectedType.toStdString()];

	showMetaType(c);
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::MetaTypeDatabaseView, mira::EditorPart);
