/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AuthorityView.h
 *    The authority view of miracenter
 *
 * @author Tim Langner, Christian Martin
 * @date   2011/01/16
 */

#ifndef _MIRA_AUTHORITYVIEW_H_
#define _MIRA_AUTHORITYVIEW_H_

#include <QTreeWidget>
#include <QPushButton>
#include <QToolButton>
#include <QTimer>

#include <fw/Framework.h>
#include <fw/MicroUnit.h>

#include <rcp/ViewPart.h>
#include <views/PropertyViewPage.h>
#include <widgets/PropertyEditor.h>
#include <widgets/TreeViewFilter.h>

#include <serialization/adapters/Qt/QAction>

#include <utils/Path.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Information about an authority
struct AuthorityItemInfo
{
	enum class RunningState {
		STARTED,
		PAUSED,
		UNKNOWN
	};

	bool complete;
	std::string id;
	bool internalError;

	RunningState isStarted;
	bool isLocal;
	bool hasUnrecoverableFailure;
	bool hasUnit;
	AbstractAuthority::ChannelNameMapping publishedMapping;
	AbstractAuthority::ChannelNameMapping subscribedMapping;
	AbstractAuthority::ChannelStatusMap channelStatus;
	std::set<std::string> interfaces;
	StatusManager::StatusMap status;

	AuthorityItemInfo() :
		complete(false), 
		internalError(false),
		isStarted(RunningState::UNKNOWN), isLocal(false),
		hasUnrecoverableFailure(false), hasUnit(false) {}
};

///////////////////////////////////////////////////////////////////////////////

/// The view class
class AuthorityView : public ViewPart
{
MIRA_META_OBJECT(AuthorityView,
                ("Name", "Authorities")
                ("Description", "Lists all available authorities"))
Q_OBJECT

public:
	AuthorityView();
	virtual ~AuthorityView();

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("ShowHidden", mShowHidden, setter<bool>(&AuthorityView::onShowHidden, this),
		           "Should hidden authorities be displayed", false);
		r.property("ShowRelative", mShowRelative, setter<bool>(&AuthorityView::onShowRelative, this),
		           "Should channels be displayed with relative names", false);
		r.property("ShowInternal", mShowInternal, setter<bool>(&AuthorityView::onShowInternal, this),
		           "Should channels be displayed with internal names", false);		           
		r.property("HideFilterBarAct",
		           getter<bool>(boost::bind(actionGetter, mHideFilterBarAct)),
		           setter<bool>(boost::bind(actionSetter, mHideFilterBarAct, _1)),
		           "Automatically hide the filter bar", false);
	}

	virtual Object* getAdapter(const Class& adapter);

	PropertyViewPage* getPropertyControl()
	{
		return mControl;
	}

	virtual Qt::DockWidgetArea getDefaultDockArea() const {
		return Qt::RightDockWidgetArea;
	}

protected:
	void updateInfoWorker();
	void updatePropertyWorker();

protected slots:

	void onShowHidden(bool show);
	void onShowRelative(bool show);
	void onShowInternal(bool show);

public:

	virtual void timerEvent(QTimerEvent *e);
	virtual QWidget* createPartControl();

	virtual bool eventFilter(QObject* obj, QEvent* event);

	void updateAuthorityTree();
	void removeItem(QTreeWidgetItem* item);
	void insertAuthority(const AuthorityItemInfo& authority);
	void updateAuthority(const AuthorityItemInfo& authority);

	void triggerUpdateProperties(const std::vector<std::string>& ids);
	void stopUpdateProperties();

private:
	AuthorityItemInfo getAuthorityInfo(const std::string authorityID);

	typedef std::map<std::string, AuthorityItemInfo> AuthorityInfoMap;

private:
	PropertyViewPage* mControl;
	AuthorityInfoMap mAuthorityInfo;
	boost::recursive_mutex mInfoMutex;
	boost::thread mInfoWorkerThread;
	boost::thread mPropertyWorkerThread;

	std::vector<std::string> mPropertyAuthorityIDs;
	boost::mutex mPropertyAuthorityIDMutex;
	boost::condition_variable mPropertyAuthorityIDCondition;
	bool mInterruptPropertiesUpdateFlag;

	std::map<std::string,QTreeWidgetItem*> mAuthorityItems;
	bool mShowHidden;
	bool mShowRelative;
	bool mShowInternal;
	QAction* mHideFilterBarAct;
	QAction* mFilterNamesAct;

	class UI;
	UI* mUI;
};

///////////////////////////////////////////////////////////////////////////////

class AuthorityView::UI : public QWidget
{
	Q_OBJECT
public:
	UI(AuthorityView* parent);
	void setupUi();

public slots:

	void itemChanged(QTreeWidgetItem* current, QTreeWidgetItem* previous);
	void itemCollapsed(QTreeWidgetItem* item);
	void itemExpanded(QTreeWidgetItem* item);
	void onSave();
	void onGenerateDocumentation();
	void updateItem(QTreeWidgetItem* item);

private slots:

	void onStart();
	void onStop();

public:
	QTreeWidget* tree;
	QToolButton* saveBtn;
	QToolButton* docBtn;
	QToolButton* startBtn;
	QToolButton* stopBtn;

	TreeViewFilter* mFilter;

private:
	void collectAuthorityIDs(QTreeWidgetItem* iItem, std::vector<std::string>& oIDVector);

	std::string getAuthorityFromItem(QTreeWidgetItem* item);

private:
	AuthorityView* mParent;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
