/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <QApplication>
#include <QPushButton>
#include <QFormLayout>
#include <QDoubleSpinBox>

#include <visualization/Visualization3DContainer.h>

#include <widgets/PropertyEditor.h>
#include <widgets/SignalBinder.h>

#include <geometry/Point.h>
#include <robot/DifferentialProbabilisticMotionModel.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

class Dialog : public QWidget
{
public:

	QFormLayout* layout;

	QDoubleSpinBox* createSpinBox(QString label, double def=0.0)
	{
		QDoubleSpinBox* sb = new QDoubleSpinBox(this);
		sb->setRange(-1.0e20,1.0e20);
		sb->setDecimals(5);
		sb->setSingleStep(0.01);
		sb->setValue(def);
		layout->addRow(label, sb);
		return sb;
	}

	Dialog() : QWidget(NULL),
		a("/","DifferentialMotionModelTest")
	{
		layout = new QFormLayout(this);

		transTransParam = createSpinBox("TransTrans",0.001);
		transRotParam = createSpinBox("TransRot",0.001);
		rotTransParam = createSpinBox("RotTrans",0.01);
		rotRotParam = createSpinBox("RotRot",0.005);
		stepParam = createSpinBox("Step",1.0);

		QPushButton* btn = new QPushButton("Test",this);
		layout->addRow(btn);
		SignalBinder::connect(btn, SIGNAL(clicked()), &Dialog::test, this);

		chParticles = a.publish<std::vector<Point2f>>("/Particles");
	}

	void closeEvent(QCloseEvent*) {
		qApp->quit();
	}


	void step(Pose2 delta)
	{
		robot::DifferentialProbabilisticMotionModel::MotionParameters p =
				model.calculateMotionParameters(Pose2(0.0f,0.0f,0.0f), delta);

		// particel filter model

		std::vector<Point2f> points(particles.size());
		for(std::size_t i=0; i<particles.size(); ++i)
		{
			particles[i] = model.sampleMotion(particles[i], p);
			points[i] = particles[i].t;
		}

		chParticles.post(points);


		// gaussian model

		//PoseCov2 ps = model.gaussianMotion(Pose2(0.0f,0.0f,0.0f), p);
		pose = model.gaussianMotion(pose, p);
		a.publishTransform<PoseCov2>("/Pose", pose, Time::now());

		// gaussian model (interval)

		poseIntervalInc = model.gaussianMotion(poseIntervalInc, p);
		poseIntervalCount++;

		if(poseIntervalCount>5) {

			std::cout << "poseInterval1: " << std::endl << poseInterval << std::endl;
			std::cout << "poseIntervalInc: " << std::endl << poseIntervalInc  << std::endl;

			poseInterval *= poseIntervalInc;

			std::cout << "poseInterval2: " << std::endl << poseInterval << std::endl;
			std::cout << "pose: " << std::endl << pose  << std::endl;

			poseIntervalInc = PoseCov2(0.0f,0.0f,0.0f, Eigen::Matrix3f::Identity()*0.0f);
			a.publishTransform<PoseCov2>("/PoseInterval", poseInterval, Time::now());
			poseIntervalCount = 0;
		}




		// compare both
		// get mean and covariance of particles:

		Eigen::Vector2f mean = Eigen::Vector2f::Zero();
		Eigen::Matrix2f cov  = Eigen::Matrix2f::Zero();

		for(std::size_t i=0; i<particles.size(); ++i)
			mean.block<2,1>(0,0) += particles[i].t;

		mean *= 1.0f / particles.size();

		for(std::size_t i=0; i<particles.size(); ++i)
		{
			Eigen::Vector2f x = particles[i].t;
			for(int j=0; j<2; ++j)
			for(int k=0; k<2; ++k)
				cov(j,k) += (x(j)-mean(j))*(x(k)-mean(k));
		}
		cov *= 1.0f / particles.size();

		/*std::cout << "Particles:" << std::endl;
		std::cout << "    Mean:" << std::endl;
		std::cout << mean << std::endl;

		std::cout << "    Cov:" << std::endl;
		std::cout << cov << std::endl;

		std::cout << std::endl;

		std::cout << "Gaussian:" << std::endl;
		std::cout << "    Mean:" << std::endl;
		std::cout << pose.t << std::endl;

		std::cout << "    Cov:" << std::endl;
		std::cout << pose.cov << std::endl;*/


	}

	void sleep(int ms)
	{
		Time end = Time::now() + Duration::milliseconds(ms);
		while(Time::now()<end)
		{
			qApp->processEvents();
			MIRA_SLEEP(10);
		}
	}

	void test()
	{
		model.alphaTransTrans = transTransParam->value();
		model.alphaTransRot = transRotParam->value();
		model.alphaRotTrans = rotTransParam->value();
		model.alphaRotRot = rotRotParam->value();

		particles.clear();
		particles.resize(1000);

		pose = PoseCov2(0.0f,-5.0f,0.0f, Eigen::Matrix3f::Identity()*0.0f);
		poseInterval = PoseCov2(0.0f,-10.0f,0.0f, Eigen::Matrix3f::Identity()*0.0f);
		poseIntervalInc = PoseCov2(0.0f,0.0f,0.0f, Eigen::Matrix3f::Identity()*0.0f);
		poseIntervalCount = 0;

		a.addTransformLink("/Pose","/GlobalFrame");
		a.addTransformLink("/PoseInterval","/GlobalFrame");

		float tscale = stepParam->value();

		for(int i=0; i<30/tscale; ++i)
		{
			step(Pose2(0.05f*tscale,0.0f,0.0f));
			sleep(50);
		}

		for(int i=0; i<10; ++i)
		{
			step(Pose2(0.0f,0.0f,half_pi<float>()/10));
			sleep(50);
		}

		for(int i=0; i<30/tscale; ++i)
		{
			step(Pose2(0.05f*tscale,0.0f,0.0f));
			sleep(50);
		}

		for(int i=0; i<10; ++i)
		{
			step(Pose2(0.0f,0.0f,-half_pi<float>()/10));
			sleep(50);
		}

		for(int i=0; i<30/tscale; ++i)
		{
			step(Pose2(0.05f*tscale,0.0f,0.0f));
			sleep(50);
		}
	}

	robot::DifferentialProbabilisticMotionModel model;

	PoseCov2 pose;
	PoseCov2 poseInterval;
	PoseCov2 poseIntervalInc;
	int poseIntervalCount;

	std::vector<Pose2> particles;



	QDoubleSpinBox* transTransParam;
	QDoubleSpinBox* transRotParam;
	QDoubleSpinBox* rotTransParam;
	QDoubleSpinBox* rotRotParam;
	QDoubleSpinBox* stepParam;

	Authority a;
	Channel<std::vector<Point2f>> chParticles;
};





int main(int argc, char** argv)
{
	QApplication qapp(argc,argv);
	Framework fw(argc,argv);

	Visualization3DContainer* widget = new Visualization3DContainer();
	PropertyEditor* e = widget->getPropertyEditor();
	e->show();
	widget->show();

	try {
		Visualization3D* v = widget->addVisualization("mira::gui::Point2fVectorVisualization3D");
		widget->setProperty(v, "PointData", "\"/Particles\"");
		widget->setProperty(v, "Size", "0.5");
		widget->setProperty(v, "DrawStyle", "3");
	} catch(Exception& ex) {
		std::cout << "This test requires the PointCloud visualization from the Maps/Mapping toolbox: " <<
				ex.message() << std::endl;
		return -1;
	}

	Visualization3D* v = widget->addVisualization("mira::gui::PoseCov2Visualization3D");
	widget->setProperty(v, "Pose", "\"/Pose\"");
	widget->setProperty(v, "Sigma", "2.0");

	v = widget->addVisualization("mira::gui::PoseCov2Visualization3D");
	widget->setProperty(v, "Pose", "\"/PoseInterval\"");


	widget->lookAt(Point3f(0,0,8),Point3f(0,-2.0f,0));
	widget->updateGL();


	Dialog* dialog = new Dialog();
	dialog->show();

	int res = qapp.exec();

	delete dialog;
	delete widget;
	return res;
}
