/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UnicycleBasedRobotModel.C
 *    Implementation of UnicycleBasedRobotModel.h.
 *
 * @author Tim Langner
 * @date   2013/02/04
 */

#include <robot/UnicycleBasedRobotModel.h>

#include <serialization/Serialization.h>
#include <serialization/DefaultInitializer.h>

///////////////////////////////////////////////////////////////////////////////

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

static float getNewVelocity(float velocity, float target, float maxAcc, float maxDec, float dt)
{
	// negative motion
	if (velocity < 0.0f)
	{
		if (target < velocity) // acceleration
			return std::max(velocity - maxAcc * dt, target); // if we accelerate to target stay at target
		else if (velocity < target)// deceleration
		{
			float newV = velocity + maxDec * dt;
			if (newV > 0 && target > 0) // if we brake to 0 accelerate for the remaining time towards target
			{
				float dt0 = -velocity / maxDec; // brake to stop
				return std::min(maxAcc * (dt - dt0), target); // if we accelerate to target stay at target
			}
			return std::min(newV, target);
		}
		return velocity;
	}
	else // positive motion
	{
		if (velocity < target) // acceleration
			return std::min(velocity + maxAcc * dt, target); // if we accelerate to target stay at target
		else if (target < velocity)// deceleration
		{
			float newV = velocity - maxDec * dt;
			if (newV < 0 && target < 0) // if we brake to 0 accelerate for the remaining time
			{
				float dt0 = velocity / maxDec; // brake to stop
				return std::max(-maxAcc * (dt - dt0), target); // if we accelerate to target stay at target
			}
			return std::max(newV, target);
		}
		return velocity;
	}
}

PoseVelocityTrajectory UnicycleBasedRobotModel::generateTrajectory(Velocity2 v, const Velocity2& targetV,
                                                                   float lookAheadTime, int samples)
{
	assert(samples>0);

	Pose2 p;
	PoseVelocityTrajectory t;

	float dt = lookAheadTime / samples;

	t.push_back(PoseVelocityTrajectorySample(p,v,0));

	for(int i=1; i<=samples; ++i)
	{
		// predict velocities
		v.x() = getNewVelocity(v.x(), targetV.x(),
		                       maxTransAcceleration, maxTransDeceleration, dt);

		v.phi() = getNewVelocity(v.phi(), targetV.phi(),
		                         maxRotAcceleration, maxRotDeceleration, dt);

		// predict pose ahead
		p = globalKinematics(v, dt, p);

		t.push_back(PoseVelocityTrajectorySample(p, v, i*dt));
	}

	return t;
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::UnicycleBasedRobotModel, mira::robot::RobotModel);
