/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GenericRobotModel.C
 *    Implementation of GenericRobotModel.h.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2011/10/21
 */

#include <robot/GenericRobotModel.h>

#include <error/Exceptions.h>
#include <serialization/Serialization.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

Velocity2 GenericRobotModel::computeRobotVelocity(const Eigen::Matrix<float,
                                                  Eigen::Dynamic,2>& castorWheelState)
{
	int wheelCount = std::min((int)wheelGeometry.size(),
	                          (int)castorWheelState.rows());

	Eigen::MatrixXf rhs(wheelCount*2,1);
	Eigen::MatrixXf lhs(wheelCount*2,3);

	// loop through all wheels
	for(int i=0; i<wheelCount; ++i)
	{
		Pose2 wheel = wheelGeometry[i];
		float phi = castorWheelState(i,1)+wheel.phi();
		float sinp = sin(phi);
		float cosp = cos(phi);

		rhs.block<2,1>(i*2,0) << castorWheelState(i,0),
		                         0;

		lhs.block<2,3>(i*2,0) << cosp, sinp, -cosp*wheel.y() + sinp*wheel.x(),
		                        -sinp, cosp,  sinp*wheel.y() + cosp*wheel.x();
	}

	// compute least square solution of robot movement in x
	Eigen::JacobiSVD<Eigen::MatrixXf> svd(lhs, Eigen::ComputeThinU | Eigen::ComputeThinV);
	Eigen::Vector3f x = svd.solve(rhs);

	return Velocity2(x.x(),x.y(),x.z());
}

Pose2 GenericRobotModel::localKinematics(const Velocity2& v, float dt) const
{
	return Pose2(v.t*dt, v.r.angle()*dt);
}

float GenericRobotModel::predictStandstillDistance(const Velocity2& v) const
{
	MIRA_THROW(XNotImplemented, "Implement me");
	return 0.0f;
}

float GenericRobotModel::predictStandstillRotation(const Velocity2& v) const
{
	MIRA_THROW(XNotImplemented, "Implement me");
	return 0.0f;
}

PoseVelocityTrajectory GenericRobotModel::generateTrajectory(Velocity2 v, const Velocity2& targetV,
                                                             float lookAheadTime, int samples) {
	MIRA_THROW(XNotImplemented, "Implement me");
	return PoseVelocityTrajectory();
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::GenericRobotModel, mira::robot::RobotModel);
