/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DriveView.C
 *    Implementation of DriveView.h.
 *
 * @author Tim Langner
 * @date   2011/04/26
 */

#include <gui/views/DriveView.h>

#include <serialization/PropertySerializer.h>
#include <transform/Velocity.h>

#include <QVBoxLayout>
#include <QMenu>
#include <QTabBar>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

DriveView::DriveView() :
	velocityIncrement(0.1f),
	watchdogTimeout(Duration::seconds(5)),
	driveService("IDrive"),
	mControl(NULL),
	ui(NULL)
{
}


DriveView::~DriveView()
{
	delete mControl;
}

QWidget* DriveView::createPartControl()
{
	ui = new UI(this);
	motorstopChannel.set("/robot/Bumper", ui);
	motorstopChannel.setDataChangedCallback(&UI::onMotorstop, ui);
	return ui;
}

void DriveView::driveServiceChanged()
{
	if (ui)
		ui->updateService(true);
}

Object* DriveView::getAdapter(const Class& adapter)
{
	if(adapter == PropertyViewPage::CLASS())
	{
		if(mControl==NULL)
		{
			PropertySerializer s;
			DriveView* This = this;
			PropertyNode* p = s.reflectProperties(getClass().getName(), This);
			mControl = new PropertyViewPage(boost::shared_ptr<PropertyNode>(p));
			mControl->setWindowTitle(this->windowTitle());
		}
		return mControl;
	}

	return ViewPart::getAdapter(adapter);
}

///////////////////////////////////////////////////////////////////////////////

DriveView::UI::UI(DriveView* parent) :
	QWidget(parent),
	mAuthority("/", "DriveView", Authority::ANONYMOUS | Authority::INTERNAL),
	mLastCommand(Time::invalid()), mParent(parent),
	mServiceQueried(false)
{
	Ui::DriveViewWidget::setupUi(this);
	setFocusPolicy(Qt::StrongFocus);
	mCommandTimer = new QTimer(this);
	mCommandTimer->setInterval(100);
	mCommandTimer->setSingleShot(false);

	mServiceTimer = new QTimer(this);
	mServiceTimer->setInterval(5000);
	mServiceTimer->setSingleShot(false);

	mBtForward->setIcon(QIcon(":/icons/ArrowUp.png"));
	mBtBackward->setIcon(QIcon(":/icons/ArrowDown.png"));
	mBtLeft->setIcon(QIcon(":/icons/ArrowLeft.png"));
	mBtRight->setIcon(QIcon(":/icons/ArrowRight.png"));
	mBtStop->setIcon(QIcon(":/icons/Cancel.png"));

	connect(mBtForward, SIGNAL(clicked()), this, SLOT(onForward()));
	connect(mBtBackward, SIGNAL(clicked()), this, SLOT(onBackward()));
	connect(mBtLeft, SIGNAL(clicked()), this, SLOT(onLeft()));
	connect(mBtRight, SIGNAL(clicked()), this, SLOT(onRight()));
	connect(mBtStop, SIGNAL(clicked()), this, SLOT(onStop()));
	connect(mBtMotorStop, SIGNAL(clicked()), this, SLOT(onResetMotorStop()));
	connect(mBtBumper, SIGNAL(clicked()), this, SLOT(onSuspendBumper()));
	connect(mBtOdometry, SIGNAL(clicked()), this, SLOT(onResetOdometry()));
	connect(mBtEnable, SIGNAL(clicked()), this, SLOT(onEnableMotors()));
	connect(mBtDisable, SIGNAL(clicked()), this, SLOT(onDisableMotors()));
	connect(mCommandTimer, SIGNAL(timeout()), this, SLOT(tick()));
	connect(mServiceTimer, SIGNAL(timeout()), this, SLOT(updateService()));
	connect(mBtGrabKeyboard, SIGNAL(toggled(bool)), this, SLOT(onGrabKeyboard(bool)));
	connect(mBtMute, SIGNAL(toggled(bool)), this, SLOT(onMute(bool)));

	updateService();
	mMute = false;
	mTransSpeed = 0;
	mRotSpeed = 0;
	mCommandTimer->start();
	mServiceTimer->start();
}

void DriveView::UI::tick()
{
	if (mMute)		// Persistent Mute
		muteAllINavigation(true);

	try {
		mParent->motorstopChannel.update();
	}
	catch(XRuntime&) {}

	if (!mLastCommand.isValid()) // do we have a command velocity?
		return;

	try {
		if ((Time::now() - mLastCommand) > mParent->watchdogTimeout)
		{
			onStop();
			return;
		}
		mAuthority.callService<void>(mService, "setVelocity",
		                             Velocity2(mTransSpeed, 0.0f, mRotSpeed));
	} catch(...) {}
}

void DriveView::UI::enableDriveButtons(bool enable)
{
	mBtForward->setEnabled(enable);
	mBtBackward->setEnabled(enable);
	mBtLeft->setEnabled(enable);
	mBtRight->setEnabled(enable);
	mBtStop->setEnabled(enable);
}

void DriveView::UI::enableMotorButtons(bool enable)
{
	mBtMotorStop->setEnabled(enable);
	mBtOdometry->setEnabled(enable);
	mBtEnable->setEnabled(enable);
	mBtDisable->setEnabled(enable);
}

void DriveView::UI::enableBumperButton(bool enable)
{
	mBtBumper->setEnabled(enable);
}

void DriveView::UI::updateService(bool serviceChanged)
{
	if (serviceChanged)
		mServiceQueried = false;

	// parent gives us a mService so use it
	if (!mParent->driveService.empty())
		mService = mParent->driveService;

	// if mService is not set in the DriveView's mServiceProperty
	// (and we don't have one yet), try to find a required interface
	// (prefer IMotorController, but also accept IDrive)
	if (mService.empty())
	{
		auto l = mAuthority.queryServicesForInterface("IMotorController");
		if (l.empty())
			l = mAuthority.queryServicesForInterface("IDrive");
		if (l.empty())
		{
			enableDriveButtons(false);
			enableMotorButtons(false);
			enableBumperButton(false);
			return;
		}
		mService = *l.begin();
	}

	if (!mAuthority.existsService(mService))
	{
		enableDriveButtons(false);
		enableMotorButtons(false);
		enableBumperButton(false);
		mService.clear();
		return;
	}

	enableDriveButtons(mAuthority.implementsInterface(mService, "IDrive"));
	enableMotorButtons(mAuthority.implementsInterface(mService, "IMotorController"));

	if (!mServiceQueried) {
		try {
			mSuspendBumperEnabledFuture = mAuthority.callService<bool>(mService, "suspendBumperSupported");
			mWaitingForSuspendBumperEnabled = true;
		}
		catch(XRPC& ex) {
			enableBumperButton(false);
		}
		mServiceQueried = true;
	}

	if (mWaitingForSuspendBumperEnabled)
		waitForSuspendBumperEnabled();
}

void DriveView::UI::waitForSuspendBumperEnabled()
{
	if (!mSuspendBumperEnabledFuture.isReady() && !mSuspendBumperEnabledFuture.hasException())
		return;

	bool enable;
	try {
		enable = mSuspendBumperEnabledFuture.get();
	}
	catch(Exception& ex) {
		enable = false;
	}

	mWaitingForSuspendBumperEnabled = false;
	enableBumperButton(enable);
}

void DriveView::UI::onForward()
{
	if (mRotSpeed == 0)
		mTransSpeed += mParent->velocityIncrement;
	mRotSpeed = 0;
	mLastCommand = Time::now();
	tick();
}

void DriveView::UI::onBackward()
{
	if (mRotSpeed == 0)
		mTransSpeed -= mParent->velocityIncrement;
	mRotSpeed = 0;
	mLastCommand = Time::now();
	tick();
}

void DriveView::UI::onLeft()
{
	mRotSpeed += mParent->velocityIncrement;
	mLastCommand = Time::now();
	tick();
}

void DriveView::UI::onRight()
{
	mRotSpeed -= mParent->velocityIncrement;
	mLastCommand = Time::now();
	tick();
}

void DriveView::UI::onStop()
{
	mRotSpeed = 0;
	mTransSpeed = 0;
	try {
		mAuthority.callService<void>(mService, "setVelocity",
		                             Velocity2(mTransSpeed, 0.0f, mRotSpeed));
		mLastCommand = Time::invalid();
	}
	catch(...) {}
}

void DriveView::UI::onResetMotorStop()
{
	mRotSpeed = 0;
	mTransSpeed = 0;
	try {
		mAuthority.callService<void>(mService, "setVelocity",
		                             Velocity2(mTransSpeed, 0.0f, mRotSpeed));
		mLastCommand = Time::invalid();
		mAuthority.callService<void>(mService, "resetMotorStop");
	}
	catch(...) {}
}

void DriveView::UI::onSuspendBumper()
{
	mRotSpeed = 0;
	mTransSpeed = 0;
	try {
		mAuthority.callService<void>(mService, "setVelocity",
		                             Velocity2(mTransSpeed, 0.0f, mRotSpeed));

		mLastCommand = Time::invalid();
		mAuthority.callService<void>(mService, "suspendBumper");
		mAuthority.callService<void>(mService, "resetMotorStop");
	}
	catch(...) {}
}

void DriveView::UI::onResetOdometry()
{
	try {
		mAuthority.callService<void>(mService, "resetOdometry");
	}
	catch(...) {}
}

void DriveView::UI::onEnableMotors()
{
	try {
			mAuthority.callService<void>(mService, "enableMotors", true);
		}
	catch(...) {}
}

void DriveView::UI::onDisableMotors()
{
	try {
			mAuthority.callService<void>(mService, "enableMotors", false);
		}
	catch(...) {}
}

void DriveView::UI::onGrabKeyboard(bool activate)
{
	if(activate) {
		this->grabKeyboard();
		getTabBar(mTabWidget)->setTabText(2, "*Input");
	} else {
		this->releaseKeyboard();
		getTabBar(mTabWidget)->setTabText(2, "Input");
	}
}

void DriveView::UI::onMute(bool mute)
{
	mMute = mute;
	muteAllINavigation(mute);

	getTabBar(mTabWidget)->setTabText(1, (mute ? "*Navigation" : "Navigation"));
}

void DriveView::UI::keyPressEvent(QKeyEvent *e)
{
	if (e->key() == Qt::Key_Right)
	{
		onRight();
		return;
	}

	if (e->key() == Qt::Key_Left)
	{
		onLeft();
		return;
	}

	if (e->key() == Qt::Key_Up)
	{
		onForward();
		return;
	}

	if (e->key() == Qt::Key_Down)
	{
		onBackward();
		return;
	}

	if (e->key() == Qt::Key_Space)
	{
		onStop();
		return;
	}


	if (e->key() == Qt::Key_Escape)
	{
		mBtGrabKeyboard->setChecked(false);
		return;
	}

	QWidget::keyPressEvent(e);
}

void DriveView::UI::onMotorstop(ChannelRead<bool> status)
{
	if (status->value()) {
		// It seems impossible to change a QPushButton's background color easily and consistently.
		// Palette is ignored by qt5ct style (but not e.g. by Windows style), while the stylesheet
		// way is over-complicated and rendered slightly differently across styles (mostly with
		// ugly result). I once liked Qt...
		//bumper->setStyleSheet("background-color:red; border:none; padding:6px");

		// So we stick to highlighting the text. Not as eye-catching, but it does as expected.
		mBtMotorStop->setStyleSheet("color:red; font-weight:bold; font-size:18px");
		getTabBar(mTabWidget)->setTabTextColor(0, Qt::red);
	} else {
		mBtMotorStop->setStyleSheet("");
		getTabBar(mTabWidget)->setTabTextColor(0, Qt::black);
	}
}

void DriveView::UI::muteAllINavigation(bool mute)
{
	auto services = mAuthority.queryServicesForInterface("INavigation");
	foreach (const auto& s, services) {
		try {
			mAuthority.callService<void>(s, "setMute", mute);
		}
		catch(...) {}
	}
}

QTabBar* DriveView::UI::getTabBar(QTabWidget* widget)
{
#if (QT_VERSION >= 0x050000)
	return mTabWidget->tabBar();
#else
	// cheap workaround:
	// https://stackoverflow.com/questions/1508658/accessing-the-qtabbar-instance
	return mTabWidget->findChild<QTabBar *>(QLatin1String("qt_tabwidget_tabbar"));
#endif
}
///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::DriveView, mira::ViewPart);
