/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Wrench.h
 *    Describing force and torque in a link
 *
 * @author Steffen Mueller
 * @date   2019/05/23
 */

#ifndef _MIRA_WRENCH_H_
#define _MIRA_WRENCH_H_

#include <math/Eigen.h>

#include <utils/IsCheapToCopy.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Wrench class.
 * Contains a force vector and a torque vector
 */
template <typename ScalarType>
class Wrench2
{
	typedef Eigen::Matrix<ScalarType,2,1> VectorType;

public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/** @name Constructors and reflect */
	//@{

	Wrench2() {}

	/**
	 * Construct a new wrench object
	 * @param[in] f Force
	 * @param[in] t Torque
	 */
	Wrench2(const VectorType& f, ScalarType t) :
		force(f),
		torque(t)
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Force", force, "The force");
		r.property("Torque", torque, "The torque");
	}

	//@}

public:
	/// The force
	VectorType force;
	/// The torque
	ScalarType torque;
};

///////////////////////////////////////////////////////////////////////////////
template <typename ScalarType>
class Wrench3
{
	typedef Eigen::Matrix<ScalarType,3,1> VectorType;

public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/** @name Constructors and reflect */
	//@{

	Wrench3() {}

	/**
	 * Construct a new wrench object
	 * @param[in] f Force
	 * @param[in] t Torque
	 */
	Wrench3(const VectorType& f, const VectorType& t) :
		force(f),
		torque(t)
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Force", force, "The force");
		r.property("Torque", torque, "The torque");
	}

	//@}

public:
	/// The force
	VectorType force;
	/// The torque
	VectorType torque;
};

/** @name Transformation operators
 * Depending on wether you want to observe a closed system from an external frame or 
 * you want to compute the forces resulting from a moment at another frame,
 * you need to add r x M to the force vector before rotating into the target frame.
 */
 //@{

/**
 * The * operator simply rotates the vectors of force and torque into another frame
 */
template<typename ScalarType>
Wrench3<ScalarType> operator * (const Pose3& trans, const Wrench3<ScalarType>& w)
{
	Eigen::Matrix<ScalarType,3,3> rot = trans.r.toRotationMatrix();
	return Wrench3<ScalarType>(rot*w.force, rot*w.torque);
}

/**
 * The ^ operator applies the translation vector of the transform with the given
 * force and computes the resulting moment in the new frame, before it transforms
 * the orientation into the new frame.
 */
template<typename ScalarType>
Wrench3<ScalarType> operator ^ (const Pose3& trans, const Wrench3<ScalarType>& w)
{
	Eigen::Matrix<ScalarType,3,3> rot = trans.r.toRotationMatrix();
	return Wrench3<ScalarType>(rot*w.force, rot*(w.torque-trans.t.cross(w.torque)));
}

//@}

/// 2D Wrench
typedef Wrench2<float> Wrench2f;
typedef Wrench2<double> Wrench2d;

/// 3D Wrench
typedef Wrench3<float> Wrench3f;
typedef Wrench3<double> Wrench3d;

///////////////////////////////////////////////////////////////////////////////

} // namespace robot

template <>
class IsCheapToCopy<robot::Wrench2f> : public std::true_type {};

template <>
class IsCheapToCopy<robot::Wrench2d> : public std::true_type {};

template <>
class IsCheapToCopy<robot::Wrench3f> : public std::true_type {};

template <>
class IsCheapToCopy<robot::Wrench3d> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif
