/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RMDF.C
 *    Implementation of RMDF.h.
 *
 * @author Tim Langner
 * @date   2011/12/05
 */

#include <boost/geometry/algorithms/correct.hpp>

#include <model/RMDF.h>

#include <transform/Pose.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

std::string loadNameFromRMDF(const XMLDom::const_iterator& rmdf)
{
	return rmdf.get_attribute<std::string>("name", "");
}

Pose3 loadOriginFromRMDF(const XMLDom::const_iterator& rmdf)
{
	Pose3 pose;
	XMLDom::const_iterator origin = rmdf.find("origin");
	if (origin != rmdf.end())
	{
		XMLDom::const_attribute_iterator xyz = origin.find_attribute("xyz");
		if (xyz != origin.attribute_end())
		{
			std::stringstream ss(xyz.value());
			ss >> pose.x() >> pose.y() >> pose.z();
		}
		XMLDom::const_attribute_iterator rpy = origin.find_attribute("rpy");
		if (rpy != origin.attribute_end())
		{
			std::stringstream ss(rpy.value());
			float r, p, y;
			ss >> r >> p >> y;
			pose.r = quaternionFromYawPitchRoll(deg2rad(y), deg2rad(p), deg2rad(r));
		}
	}
	return pose;
}

Point3f loadAxisFromRMDF(const XMLDom::const_iterator& rmdf)
{
	Point3f point(1.0f, 0.0f, 0.0f);
	XMLDom::const_iterator axis = rmdf.find("axis");
	if (axis != rmdf.end())
	{
		XMLDom::const_attribute_iterator xyz = axis.find_attribute("xyz");
		if (xyz != axis.attribute_end())
		{
			std::stringstream ss(xyz.value());
			ss >> point.x() >> point.y() >> point.z();
		}
	}
	return point;
}

///////////////////////////////////////////////////////////////////////////////

MaterialPtr loadMaterialFromRMDF(const XMLDom::const_iterator& rmdf)
{
	MaterialPtr material(new Material());
	material->name = rmdf.get_attribute<std::string>("name");
	bool hasContent = false;
	// optional color
	XMLDom::const_iterator colorTag = rmdf.find("color");
	if (colorTag != rmdf.end())
	{
		hasContent = true;
		std::stringstream ss(colorTag.get_attribute<std::string>("rgba"));
		ss >> material->color.r >> material->color.g >> material->color.b >> material->color.a;
	}
	else
		material->color = Color::RGBA(1.0f, 1.0f, 1.0f, 1.0f);
	// optional texture
	XMLDom::const_iterator textureTag = rmdf.find("texture");
	if (textureTag != rmdf.end())
	{
		hasContent = true;
		material->texture = textureTag.get_attribute<std::string>("filename");
	}
	else
		material->texture = "";
	return hasContent?material:MaterialPtr();
}

///////////////////////////////////////////////////////////////////////////////

GeometryPtr loadGeometryFromRMDF(const XMLDom::const_iterator& rmdf)
{
	XMLDom::const_iterator geometry = rmdf.find("geometry");
	if (geometry == rmdf.end())
		MIRA_THROW(XInvalidConfig, "No 'geometry' tag found in node '" << *rmdf
		           << "' line=" << rmdf.line() << " file=" << rmdf.uri());
	XMLDom::const_iterator shape = geometry.begin();
	if (*shape == "box")
	{
		BoxPtr box(new Box());
		std::stringstream ss(shape.get_attribute<std::string>("size"));
		ss >> box->size.width() >> box->size.height() >> box->size.depth();
		return box;
	}
	else if (*shape == "cone")
	{
		ConePtr cone(new Cone());
		cone->radius = shape.get_attribute<float>("radius");
		cone->length = shape.get_attribute<float>("length");
		return cone;
	}
	else if (*shape == "cylinder")
	{
		CylinderPtr cylinder(new Cylinder());
		cylinder->radius = shape.get_attribute<float>("radius");
		cylinder->length = shape.get_attribute<float>("length");
		return cylinder;
	}
	else if (*shape == "sphere")
	{
		SpherePtr sphere(new Sphere());
		sphere->radius = shape.get_attribute<float>("radius");
		return sphere;
	}
	else if (*shape == "mesh")
	{
		MeshPtr mesh(new Mesh());
		mesh->filename = shape.get_attribute<std::string>("filename");
		mesh->scale = shape.get_attribute<float>("scale", 1.0f);
		return mesh;
	}
	else if (*shape == "polygon")
	{
		PolygonPtr poly(new Polygon());
		XMLDom::const_iterator point = shape.begin();
		for ( ; point != shape.end(); ++point)
		{
			if (*point == "point")
			{
				Point2f p;
				std::stringstream ss(point.get_attribute<std::string>("xy"));
				ss >> p.x() >> p.y();
				poly->polygon.push_back(p);
			}
		}
		boost::geometry::correct(poly->polygon);
		return poly;
	}
	else
		MIRA_THROW(XInvalidConfig, "Unknown shape '" << *shape << "'");
	return GeometryPtr();
}

///////////////////////////////////////////////////////////////////////////////

InertialPtr loadInertialFromRMDF(const XMLDom::const_iterator& rmdf)
{
	InertialPtr inertial(new Inertial());
	// optional origin
	inertial->origin = loadOriginFromRMDF(rmdf);
	return inertial;
}

///////////////////////////////////////////////////////////////////////////////

CollisionPtr loadCollisionFromRMDF(const XMLDom::const_iterator& rmdf)
{
	CollisionPtr collision(new Collision());
	// optional name
	collision->name = loadNameFromRMDF(rmdf);
	// optional origin
	collision->origin = loadOriginFromRMDF(rmdf);
	// required geometry
	collision->geometry = loadGeometryFromRMDF(rmdf);
	return collision;
}

///////////////////////////////////////////////////////////////////////////////

VisualPtr loadVisualFromRMDF(const XMLDom::const_iterator& rmdf)
{
	VisualPtr visual(new Visual());
	// optional name
	visual->name = loadNameFromRMDF(rmdf);
	// optional origin
	visual->origin = loadOriginFromRMDF(rmdf);
	// required geometry
	visual->geometry = loadGeometryFromRMDF(rmdf);
	// optional material
	XMLDom::const_iterator materialTag = rmdf.find("material");
	if (materialTag != rmdf.end())
	{
		visual->materialName = materialTag.get_attribute<std::string>("name");
		visual->material = loadMaterialFromRMDF(materialTag);
	}
	return visual;
};

///////////////////////////////////////////////////////////////////////////////

void copyToCollision(VisualPtr from, CollisionPtr to)
{
	to->name = from->name;
	to->origin = from->origin;
	to->geometry = from->geometry;
}

LinkPtr loadLinkFromRMDF(const XMLDom::const_iterator& rmdf)
{
	LinkPtr link(new Link());
	link->name = rmdf.get_attribute<std::string>("name");
	// optional inertial
	XMLDom::const_iterator inertialTag = rmdf.find("inertial");
	if (inertialTag != rmdf.end())
		link->inertial = loadInertialFromRMDF(inertialTag);

	XMLDom::const_iterator vi = rmdf.begin();
	for ( ; vi != rmdf.end(); ++vi)
		if (*vi == "visual")
			link->visuals.push_back(loadVisualFromRMDF(vi));

	// optional collision
	XMLDom::const_iterator ci = rmdf.begin();
	for ( ; ci != rmdf.end(); ++ci)
	{
		if (*ci == "collision")
		{
			// EXTENSION begin
			XMLDom::const_attribute_iterator like = ci.find_attribute("like");
			if (like != ci.attribute_end())
			{
				VisualPtr from;
				// search for visual with matching name
				foreach(VisualPtr v, link->visuals)
					if (v->name == like.value())
					{
						from = v;
						break;
					}
				// if no match found and value is visual -> just use first visual as reference
				if (!from && like.value()=="visual" && !link->visuals.empty())
					from = link->visuals[0];

				// check if reference was found
				if (!from)
					MIRA_THROW(XIO, "No visual defined for like='" << like.value() << "' attribute in link '"<< link->name <<"'");

				CollisionPtr collision(new Collision());
				copyToCollision(from, collision);
				link->collisions.push_back(collision);
			}
			// EXTENSION
			else
				link->collisions.push_back(loadCollisionFromRMDF(ci));
		}
	}
	return link;
}

///////////////////////////////////////////////////////////////////////////////

JointPtr loadJointFromRMDF(const XMLDom::const_iterator& rmdf)
{
	JointPtr joint(new Joint());
	joint->name = rmdf.get_attribute<std::string>("name");
	std::string type = rmdf.get_attribute<std::string>("type");
	if (type == "revolute")
		joint->type = Joint::REVOLUTE;
	else if (type == "continuous")
		joint->type = Joint::CONTINUOUS;
	else if (type == "prismatic")
		joint->type = Joint::PRISMATIC;
	else if (type == "fixed")
		joint->type = Joint::FIXED;
	else if (type == "floating")
		joint->type = Joint::FLOATING;
	else
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has invalid type");
	// optional origin
	joint->origin = loadOriginFromRMDF(rmdf);
	// required parent
	XMLDom::const_iterator parentTag = rmdf.find("parent");
	if (parentTag == rmdf.end())
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has no parent tag");
	joint->parent = parentTag.get_attribute<std::string>("link");
	// required child
	XMLDom::const_iterator childTag = rmdf.find("child");
	if (childTag == rmdf.end())
		MIRA_THROW(XInvalidConfig, "Joint '" << joint->name << "' has no child tag");
	joint->child = childTag.get_attribute<std::string>("link");
	// optional axis
	joint->axis = loadAxisFromRMDF(rmdf);
	// optional limit if continuous, floating or fixed
	XMLDom::const_iterator limitTag = rmdf.find("limit");
	if (limitTag == rmdf.end() && 
		(joint->type == Joint::REVOLUTE || joint->type == Joint::PRISMATIC))
		MIRA_THROW(XInvalidConfig, "Missing limit tag for revolute/prismatic joint '" << joint->name << "'");
	if (limitTag != rmdf.end())
	{
		// check for lower limit of the joint
		if (limitTag.find_attribute("lower") != limitTag.attribute_end())
		{
			float value = limitTag.get_attribute<float>("lower");
			// revolute joints have angular limit (degree) 
			if (joint->type == Joint::REVOLUTE)
				value = deg2rad(value);
			joint->lowerLimit = value;
		}
		// check for upper limit of the joint
		if (limitTag.find_attribute("upper") != limitTag.attribute_end())
		{
			float value = limitTag.get_attribute<float>("upper");
			// revolute joints have angular limit (degree) 
			if (joint->type == Joint::REVOLUTE)
				value = deg2rad(value);
			joint->upperLimit = value;
		}
	}
	// TODO calibration, dynamics,...
	return joint;
}

///////////////////////////////////////////////////////////////////////////////

void loadModelFromRMDF(const XMLDom::const_iterator& rmdf, RigidModel& oModel)
{
	oModel.clear();

	std::set<std::string> linkNames;

	// needs to start with a <model name="name"> tag
	if (*rmdf != "model")
		MIRA_THROW(XInvalidConfig, "RMDF does not contain 'model' root node");
	oModel.name = rmdf.get_attribute<std::string>("name");
	// iterate over all child nodes
	XMLDom::const_iterator i = rmdf.begin();
	for ( ; i != rmdf.end(); ++i)
	{
		// materials are allowed to be top level nodes
		if (*i == "material")
		{
			MaterialPtr m = loadMaterialFromRMDF(i);
			if (oModel.materials.count(m->name) != 0)
				MIRA_THROW(XInvalidConfig, "Material must be unique ('" << m->name << "')");
			oModel.materials[m->name] = m;
		}
		// links (frames) are allowed to be top level nodes
		else if (*i == "link")
		{
			LinkPtr l = loadLinkFromRMDF(i);
			if (!linkNames.insert(l->name).second)
				MIRA_THROW(XInvalidConfig, "Link must be unique ('" << l->name << "')");
			oModel.links.push_back(l);
			// iterate over all visual components of the link and look for materials
			// that are either named references to a global material or
			// are new materials that get inserted into the global material list
			// checks also for undefined materials
			foreach(VisualPtr v, l->visuals)
			{
				// if visual has a material
				if (v && !v->materialName.empty())
				{
					// check if its already contained in the global material list
					if (oModel.materials.count(v->materialName))
					{
						// reference to the global material
						v->material = oModel.materials.find(v->materialName)->second;
					}
					else // otherwise its a new one
					{
						// if the material is fully defined (not just a name reference)
						// add a reference to the global material list
						if (v->material)
							oModel.materials[v->materialName] = v->material;
						else // material is a reference to an undefined material
							MIRA_THROW(XInvalidConfig, "Material '" << v->materialName << "' is undefined");
					}
				}
			}
		}
		// joints are allowed to be top level nodes
		else if (*i == "joint")
		{
			JointPtr j = loadJointFromRMDF(i);
			if (oModel.joints.count(j->name) != 0)
				MIRA_THROW(XInvalidConfig, "Joint must be unique ('" << j->name << "')");
			oModel.joints[j->name] = j;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}}
