/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Geometry.C
 *    Implementation of Geometry.h.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2011/05/09
 */

#include <boost/version.hpp>

#if BOOST_VERSION < 105600
#include <boost/geometry/multi/geometries/multi_point.hpp>
#else
#include <boost/geometry/geometries/multi_point.hpp>
#endif
#include <boost/geometry/algorithms/convex_hull.hpp>


#include <boost/geometry/geometries/polygon.hpp>
#include <boost/geometry/strategies/strategies.hpp>

#include <model/Geometry.h>

#include <serialization/Serialization.h>
#include <transform/TransformCast.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

template <typename PointContainer>
void pointsToFootprint(Polygon2f& oFootprint,
                       PointContainer& points, int pointsCount,
                       const RigidTransform3f& origin)
{
	Polygon2f v;
	for(int i=0; i<pointsCount; ++i)
	{
		Point3f tp = origin * points[i];
		// project to 2D plane and add to polygon
		v.push_back(Point2f(tp.x(), tp.y()));
	}

	boost::geometry::convex_hull(v, oFootprint);
}

template <typename PointContainer>
void pointsToBoundingBox(Box3f& oBox,
                         PointContainer& points, int pointsCount,
                         const RigidTransform3f& origin)
{
	Box3f v = Box3f::invalid();
	for(int i=0; i<pointsCount; ++i)
	{
		Point3f tp = origin * points[i];
		v |= tp;
	}
	oBox = v;
}

///////////////////////////////////////////////////////////////////////////////

inline Point3f* boxPoints(Point3f* p, const Size3f& size)
{
	Size3f  hsize = size * 0.5f;
	p[0] = Point3f(-hsize.x(),-hsize.y(),-hsize.z());
	p[1] = Point3f(+hsize.x(),-hsize.y(),-hsize.z());
	p[2] = Point3f(-hsize.x(),+hsize.y(),-hsize.z());
	p[3] = Point3f(+hsize.x(),+hsize.y(),-hsize.z());
	p[4] = Point3f(-hsize.x(),-hsize.y(),+hsize.z());
	p[5] = Point3f(+hsize.x(),-hsize.y(),+hsize.z());
	p[6] = Point3f(-hsize.x(),+hsize.y(),+hsize.z());
	p[7] = Point3f(+hsize.x(),+hsize.y(),+hsize.z());
	return p;
}

Polygon2f Box::getFootprint(const RigidTransform3f& origin) const
{
	Point3f p[8];
	boxPoints(p, size);
	Polygon2f polygon;
	pointsToFootprint(polygon, p, 8, origin);
	return polygon;
}

Box3f Box::getBoundingBox(const RigidTransform3f& origin) const
{
	Point3f p[8];
	boxPoints(p, size);
	Box3f box;
	pointsToBoundingBox(box, p, 8, origin);
	return box;
}

///////////////////////////////////////////////////////////////////////////////

// precision for spheres, cones and cylinders
static const int sectors = 24;


void conePoints(std::vector<Point3f>& oV, float radius, float length)
{
	float alpha=0;
	float dalpha = two_pi<float>() / sectors;
	for(int i=0; i<=sectors; ++i, alpha+=dalpha)
		oV.push_back( Point3f(sin(alpha)*radius, cos(alpha)*radius, 0) );
	// apex
	oV.push_back( Point3f(0, 0, length) );
}

Polygon2f Cone::getFootprint(const RigidTransform3f& origin) const
{
	std::vector<Point3f> v;
	conePoints(v, radius, length);
	Polygon2f polygon;
	pointsToFootprint(polygon, v, v.size(), origin);
	return polygon;
}

Box3f Cone::getBoundingBox(const RigidTransform3f& origin) const
{
	std::vector<Point3f> v;
	conePoints(v, radius, length);
	Box3f box;
	pointsToBoundingBox(box, v, v.size(), origin);
	return box;
}

///////////////////////////////////////////////////////////////////////////////

void cylinderPoints(std::vector<Point3f>& oV, float radius, float length)
{
	float alpha=0;
	float dalpha = two_pi<float>() / sectors;
	for(int i=0; i<=sectors; ++i, alpha+=dalpha) {
		oV.push_back( Point3f(sinf(alpha)*radius, cosf(alpha)*radius, -length*0.5f) );
		oV.push_back( Point3f(sinf(alpha)*radius, cosf(alpha)*radius, length*0.5f) );
	}
}

Polygon2f Cylinder::getFootprint(const RigidTransform3f& origin) const
{
	std::vector<Point3f> v;
	cylinderPoints(v, radius, length);
	Polygon2f polygon;
	pointsToFootprint(polygon, v, v.size(), origin);
	return polygon;
}

Box3f Cylinder::getBoundingBox(const RigidTransform3f& origin) const
{
	std::vector<Point3f> v;
	cylinderPoints(v, radius, length);
	Box3f box;
	pointsToBoundingBox(box, v, v.size(), origin);
	return box;
}

///////////////////////////////////////////////////////////////////////////////

Polygon2f Sphere::getFootprint(const RigidTransform3f& origin) const
{
	Point2f c(origin.x(),origin.y());
	Polygon2f polygon;

	float alpha=0;
	float dalpha = two_pi<float>() / sectors;
	for(int i=0; i<=sectors; ++i, alpha+=dalpha)
	{
		Point2f d(sin(alpha)*radius, cos(alpha)*radius);
		polygon.push_back(c+d);
	}
	return polygon;
}

Box3f Sphere::getBoundingBox(const RigidTransform3f& origin) const
{
	Point3f r(radius,radius,radius);
	return Box3f(Point3f(origin.t-r), Point3f(origin.t+r));
}

///////////////////////////////////////////////////////////////////////////////

Polygon2f Mesh::getFootprint(const RigidTransform3f& origin) const
{
	MIRA_THROW(XNotImplemented, "Mesh::getFootprint() not implemented.");

	// TODO: implement me (see #397)
	return Polygon2f();
}

Box3f Mesh::getBoundingBox(const RigidTransform3f& origin) const
{
	MIRA_THROW(XNotImplemented, "Mesh::getBoundingBox() not implemented.");

	// TODO: implement me (see #397)
	return Box3f::invalid();
}

///////////////////////////////////////////////////////////////////////////////

Polygon2f Polygon::getFootprint(const RigidTransform3f& origin) const
{
	Polygon2f tpolygon; // the transformed polygon

	RigidTransform2f origin2 = transform_cast<RigidTransform2f>(origin);

	// transform each point of the polygon:
	foreach(const Point2f& p, polygon)
	{
		Point2f tp = origin2 * p;
		tpolygon.push_back(tp);
	}

	return tpolygon;
}

Box3f Polygon::getBoundingBox(const RigidTransform3f& origin) const
{
	// the polygon is 2D, hence its bounding box would be 2D too, which is
	// still empty in 3D, hence we can return an invalid empty box
	return Box3f::invalid();
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_REGISTER(mira::model::Geometry, mira::Object);
MIRA_CLASS_SERIALIZATION(mira::model::Box, mira::model::Geometry);
MIRA_CLASS_SERIALIZATION(mira::model::Cone, mira::model::Geometry);
MIRA_CLASS_SERIALIZATION(mira::model::Cylinder, mira::model::Geometry);
MIRA_CLASS_SERIALIZATION(mira::model::Sphere, mira::model::Geometry);
MIRA_CLASS_SERIALIZATION(mira::model::Polygon, mira::model::Geometry);
MIRA_CLASS_SERIALIZATION(mira::model::Mesh, mira::model::Geometry);
