/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JointPublisherView.C
 *    Implementation of JointPublisherView.h.
 *
 * @author Tim Langner
 * @date   2011/12/08
 */

#include <gui/views/JointPublisherView.h>

#include <QVariant>
#include <QLabel>
#include <QHBoxLayout>
#include <QDoubleValidator>

#include <math/Angle.h>

#define SLIDER_STEPS 10000

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

JointPublisherView::JointPublisherView() :
	mControl(NULL),
	ui(NULL)
{
}

JointPublisherView::~JointPublisherView()
{
	delete mControl;
}

QWidget* JointPublisherView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

Object* JointPublisherView::getAdapter(const Class& adapter)
{
	if(adapter == PropertyViewPage::CLASS())
	{
		if(mControl==NULL)
		{
			PropertySerializer s;
			JointPublisherView* This = this;
			PropertyNode* p = s.reflectProperties(getClass().getName(), This);
			mControl = new PropertyViewPage(boost::shared_ptr<PropertyNode>(p));
			mControl->setWindowTitle(this->windowTitle());
		}
		return mControl;
	}

	return ViewPart::getAdapter(adapter);
}


JointPublisherView::UI::UI(JointPublisherView* parent) :
	QWidget(parent),
	authority("/", "JointPublisherView", Authority::ANONYMOUS | Authority::INTERNAL),
	mParent(parent)
{
	mLayout = new QVBoxLayout(this);

	serviceTimer = new QTimer(this);
	serviceTimer->setInterval(2000);
	serviceTimer->setSingleShot(false);
		
	connect(serviceTimer, SIGNAL(timeout()), this, SLOT(updateService()));

	updateService();
	serviceTimer->start();
}

void JointPublisherView::UI::updateService()
{
	// parent gives us a service so use it
	if (!mParent->modelProvider.empty())
	{
		if (mParent->modelProvider != service)
		{
			service = mParent->modelProvider;
			setupModel();
		}
		return;
	}

	if (service.empty())
	{
		auto l = authority.queryServicesForInterface("IRigidModelProvider");
		if (l.empty())
		{
			clearModel();
			return;
		}
		service = *l.begin();
		setupModel();
	}
}

void clearLayoutChildren(QLayout* l)
{
	QLayoutItem* child;
	while ((child = l->takeAt(0)) != NULL)
	{
		if (child->layout() != NULL)
		{
			clearLayoutChildren(child->layout());
			child->layout()->deleteLater();
		}
		else
			child->widget()->deleteLater();
		delete child;
	}
}

void JointPublisherView::UI::clearModel()
{
	mJointInfos.clear();
	clearLayoutChildren(mLayout);
}

int valueToSlider(float value, float min, float max)
{
	return (int)((value - min) * float(SLIDER_STEPS) / (max - min));
}

float sliderToValue(int value, float min, float max)
{
	float factor = value / float(SLIDER_STEPS);
	return min + (max-min) * factor;
}

void JointPublisherView::UI::setupModel()
{
	clearModel();
	auto p = authority.callService<RigidModelPtr>(service, "getRigidModel");
	if (!p.timedWait(Duration::seconds(1000)))
		MIRA_THROW(XRuntime, "Model could not be retrieved from service '" <<
				service << "'");

	model = p.get();
	assert(model);

	foreach(const auto& j, model->joints)
	{
		if (j.second->type == Joint::REVOLUTE || 
			j.second->type == Joint::PRISMATIC ||
			j.second->type == Joint::CONTINUOUS)
		{
			JointInfo info;
			QVBoxLayout* vLayout = new QVBoxLayout();
			QHBoxLayout* hLayout = new QHBoxLayout();
			QLabel* label = new QLabel(j.first.c_str(), this);
			info.edit = new QLineEdit(this);
			info.edit->setMaximumWidth(60);
			info.edit->setProperty("Joint", QVariant(QString(j.first.c_str())));
			info.slider = new QSlider(Qt::Horizontal, this);
			info.slider->setMinimum(0);
			info.slider->setMaximum(SLIDER_STEPS);
			info.slider->setSingleStep(1);
			info.slider->setProperty("Joint", QVariant(QString(j.first.c_str())));

			float start;
			if (j.second->type == Joint::REVOLUTE)
			{
				info.min = rad2deg(j.second->lowerLimit);
				info.max = rad2deg(j.second->upperLimit);
				connect(info.slider, SIGNAL(sliderMoved(int)), this,
						SLOT(revolute(int)));
				connect(info.edit, SIGNAL(textEdited(const QString&)), this,
						SLOT(revolute(const QString&)));
			}
			else if (j.second->type == Joint::CONTINUOUS)
			{
				info.min = -180.0f;
				info.max = 180.0f;
				connect(info.slider, SIGNAL(sliderMoved(int)), this,
						SLOT(revolute(int)));
				connect(info.edit, SIGNAL(textEdited(const QString&)), this,
						SLOT(revolute(const QString&)));
			}
			else
			{
				info.min = j.second->lowerLimit;
				info.max = j.second->upperLimit;
				connect(info.slider, SIGNAL(sliderMoved(int)), this,
						SLOT(prismatic(int)));
				connect(info.edit, SIGNAL(textEdited(const QString&)), this,
						SLOT(prismatic(const QString&)));
			}
			info.edit->setValidator(new QDoubleValidator(info.min, info.max, 2, info.edit));
			mJointInfos[j.first] = info;
			start = info.min + (info.max-info.min)/2;
			info.edit->setText(QString("%1").arg(start));
			info.slider->setValue(valueToSlider(start, info.min, info.max));
			hLayout->addWidget(label);
			hLayout->addStretch();
			hLayout->addWidget(info.edit);
			vLayout->addLayout(hLayout);
			vLayout->addWidget(info.slider);
			mLayout->addLayout(vLayout);
		}
	}
}

void JointPublisherView::UI::revolute(int value)
{
	QSlider* slider = (QSlider*)sender();
	std::string joint = slider->property("Joint").toString().toStdString();
	JointInfo info = mJointInfos[joint];
	float v = sliderToValue(value, info.min, info.max);
	model->publishJoint(joint, deg2rad(v));
	info.edit->setText(QString("%1").arg(v));
}

void JointPublisherView::UI::prismatic(int value)
{
	QSlider* slider = (QSlider*)sender();
	std::string joint = slider->property("Joint").toString().toStdString();
	JointInfo info = mJointInfos[joint];
	float v = sliderToValue(value, info.min, info.max);
	model->publishJoint(joint, v);
	info.edit->setText(QString("%1").arg(v));
}

void JointPublisherView::UI::revolute(const QString& value)
{
	QLineEdit* edit = (QLineEdit*)sender();
	std::string joint = edit->property("Joint").toString().toStdString();
	JointInfo info = mJointInfos[joint];
	float v = value.toFloat();
	model->publishJoint(joint, deg2rad(v));
	info.slider->setValue(valueToSlider(v, info.min, info.max));
}

void JointPublisherView::UI::prismatic(const QString& value)
{
	QLineEdit* edit = (QLineEdit*)sender();
	std::string joint = edit->property("Joint").toString().toStdString();
		JointInfo info = mJointInfos[joint];
		float v = value.toFloat();
	model->publishJoint(joint, v);
	info.slider->setValue(valueToSlider(v, info.min, info.max));
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::model::JointPublisherView, mira::ViewPart);
