/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LocalizeTool2D.C
 *    Tool to specify poses for a ILocalization service in the 2D graphics view.
 *
 * @author Christian Reuther
 * @date   2014/08/29
 */

#include <visualization/2d/PoseTool2D.h>

#include <fw/ServiceProperty.h>

namespace mira { namespace localization {

///////////////////////////////////////////////////////////////////////////////

class LocalizeTool2D : public PoseTool2D
{
	MIRA_META_OBJECT(LocalizeTool2D,
	                 ("Name", "Localize")
	                 ("Description", "Specifies poses for a ILocalization service")
	                 ("Category", "Localization")
	                 ("Order", "100"))

public:
	LocalizeTool2D() : mLocalizationService("ILocalization") {}
	virtual ~LocalizeTool2D() {}

	template<typename Reflector> void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, PoseTool2D);

		r.property("LocalizationService", mLocalizationService, "Service to call setInitPose for.", "");
	}

	virtual QIcon getIcon();

protected:
	virtual void onNewPose(const PoseCov2& pose);
	std::string getService();

	ServiceProperty mLocalizationService;
};

///////////////////////////////////////////////////////////////////////////////

QIcon LocalizeTool2D::getIcon() {
	return QIcon(QPixmap(":/icons/ToolLocalize.png"));
}

void LocalizeTool2D::onNewPose(const PoseCov2& pose) {
	// First try with user-defined service. If not available, search for one.
	const std::string service = getService();
	if(service.empty())
		return;

	auto ftr = callService<void>(service, "setInitPose", pose, getSite()->getFixedFrame());
	ftr.timedWait(Duration::seconds(10));
	ftr.get();
}

std::string LocalizeTool2D::getService() {
	std::string s = mLocalizationService;
	// user did not provide a service -> search for interface
	if(s.empty()) {
		auto services = queryServicesForInterface("ILocalization");
		s = services.empty() ? "" : services.front();
	}
	return s;
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::localization::LocalizeTool2D, mira::VisualizationTool2D)
