/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformPropertyDelegate.C
 *    A delegate for transform properties.
 *
 * @author Erik Einhorn
 * @date   2011/10/01
 */

#include <boost/format.hpp>

#include <transform/RigidTransform.h>

#include <widgets/PropertyEditor.h>

#include <serialization/PropertySerializer.h>

///////////////////////////////////////////////////////////////////////////////

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

float rad2degNorm(float value)
{
	return Radianf(value).deg();
}

std::string transformToString(const RigidTransform2f& t) {
	return (boost::format("[%.3f, %.3f] %.3f") % t.x() % t.y() %
	        rad2degNorm(t.phi())).str();
}

std::string transformToString(const RigidTransform3f& t) {
	return (boost::format("[%.3f, %.3f, %f] %.3f, %.3f, %.3f")
	        % t.x() % t.y() % t.z()
	        % rad2degNorm(t.yaw()) % rad2degNorm(t.pitch())
	        % rad2degNorm(t.roll())).str();
}

//////////////////////////////////////////////////////////////////////////////

template <typename Transform>
class PropertyDelegateTransformBase : public PropertyEditor::Delegate
{
public:

	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<Transform>(SupportedType::TEXT);
	}

	virtual std::string getText(const PropertyNode* property) {
		const TypedPropertyNode<Transform>* p = property->toTyped<Transform>();
		Transform t = p->get();
		return transformToString(t);
	}
};

class PropertyDelegateTransform2f :
	public  PropertyDelegateTransformBase<RigidTransform2f>
{
	MIRA_OBJECT(PropertyDelegateTransform2f)
};

class PropertyDelegateTransform3f :
	public PropertyDelegateTransformBase<RigidTransform3f>
{
	MIRA_OBJECT(PropertyDelegateTransform3f)
};

class PropertyDelegateTransformCov2f :
	public PropertyDelegateTransformBase<RigidTransformCov2f>
{
	MIRA_OBJECT(PropertyDelegateTransformCov2f)
};

class PropertyDelegateTransformCov3f :
	public PropertyDelegateTransformBase<RigidTransformCov3f>
{
	MIRA_OBJECT(PropertyDelegateTransformCov3f)
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_REGISTER(mira::gui::PropertyDelegateTransform2f,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegateTransform3f,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegateTransformCov2f,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegateTransformCov3f,
                    mira::PropertyEditor::Delegate);
