/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/*
 * @file BoxVisualization
 *    Visualization of a vector of boxes.
 *
 * @author: Jens Kessler, Erik Einhorn
 * @date:   2011/03/01
 */

#include <OGRE/OgreManualObject.h>

#include <serialization/DefaultInitializer.h>
#include <serialization/SetterNotify.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>


#include <geometry/Rect.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/LineStripObject.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////


class BoxRenderer
{

public:

	BoxRenderer() : mSceneManager(NULL), mNode(NULL), mLineObject(NULL)
	{
		MIRA_INITIALIZE_THIS;
		mEnabled = false;
	}

	~BoxRenderer()
	{
		if (mSceneManager == NULL)
			return;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Color", mColor, "The color of the boxes",
				   Ogre::ColourValue::Red);
		r.property("Line Width", mLineWidth, setterNotify(mLineWidth, &BoxRenderer::updateLineWidth, this),
		           "Line Width", 0.0f, PropertyHints::step(0.02f));
	}


	void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mNode = node;
		mSceneManager = mgr;
		mLineObject = new LineStripObject(mgr, node);
		mLineObject->setLineWidth(mLineWidth);
	}


	void updateLineWidth()
	{
		if(mLineObject==NULL) // not yet initialized
			return;

		mLineObject->setLineWidth(mLineWidth);
		setEnabled(mEnabled);
	}

	void setEnabled(bool enabled) {
		mEnabled = enabled;
		if (mNode != NULL) {
			mNode->setVisible(enabled);
			mLineObject->setVisible(enabled);
		}
	}

	void clear()
	{
		mLineObject->clear();
		mLineObject->setColor(mColor);
	}


	void drawLine(float x1, float y1, float z1, float x2, float y2, float z2)
	{
		mLineObject->begin();
		mLineObject->point(Ogre::Vector3(x1,y1,z1));
		mLineObject->point(Ogre::Vector3(x2,y2,z2));
		mLineObject->end();

	}
	void drawBox(const Box3f& b)
	{
		drawLine(b.x0(),b.y0(), b.z0(),  b.x0(),b.y0(), b.z1());
		drawLine(b.x1(),b.y0(), b.z0(),  b.x1(),b.y0(), b.z1());
		drawLine(b.x0(),b.y1(), b.z0(),  b.x0(),b.y1(), b.z1());
		drawLine(b.x1(),b.y1(), b.z0(),  b.x1(),b.y1(), b.z1());

		drawLine(b.x0(),b.y0(), b.z0(),  b.x0(),b.y1(), b.z0());
		drawLine(b.x1(),b.y0(), b.z0(),  b.x1(),b.y1(), b.z0());
		drawLine(b.x0(),b.y0(), b.z1(),  b.x0(),b.y1(), b.z1());
		drawLine(b.x1(),b.y0(), b.z1(),  b.x1(),b.y1(), b.z1());

		drawLine(b.x0(),b.y0(), b.z0(),  b.x1(),b.y0(), b.z0());
		drawLine(b.x0(),b.y1(), b.z0(),  b.x1(),b.y1(), b.z0());
		drawLine(b.x0(),b.y0(), b.z1(),  b.x1(),b.y0(), b.z1());
		drawLine(b.x0(),b.y1(), b.z1(),  b.x1(),b.y1(), b.z1());

	}
private:
	bool mEnabled;
	Ogre::SceneManager* mSceneManager;
	Ogre::SceneNode* mNode;
	Ogre::ColourValue mColor;
	float mLineWidth;
	LineStripObject* mLineObject;
};


class BoxVectorVisualization3D : public Visualization3DBasic<std::vector<Box3f>>
{
	MIRA_META_OBJECT(BoxVectorVisualization3D,
		("Name", "Boxes")
		("Category", "Geometry")
		("Description", "Shows a vector of boxes"))

public:

	BoxVectorVisualization3D() :
		Visualization3DBasic<std::vector<Box3f>>("Boxes")
	{}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3DBasic<std::vector<Box3f>>);
		mRenderer.reflect(r);
	}

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) {
		mRenderer.setupScene(mgr,node);
	}

	void setEnabled(bool enabled) {
		mRenderer.setEnabled(enabled);
	}

protected:

	void dataChanged(ChannelRead<std::vector<Box3f>> boxes)
	{
		mRenderer.clear();
		foreach(const Box3f& b, boxes->value())
			mRenderer.drawBox(b);
	}
private:
	BoxRenderer mRenderer;
};


class BoxVisualization3D : public Visualization3DBasic<Box3f>
{
	MIRA_META_OBJECT(BoxVisualization3D,
		("Name", "Box")
		("Category", "Geometry")
		("Description", "Shows a box"))

public:

	BoxVisualization3D() :
		Visualization3DBasic<Box3f>("Box")
	{}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3DBasic<Box3f>);
		mRenderer.reflect(r);
	}

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node) {
		mRenderer.setupScene(mgr,node);
	}

	void setEnabled(bool enabled) {
		mRenderer.setEnabled(enabled);
	}

protected:

	void dataChanged(ChannelRead<Box3f> d)
	{
		mRenderer.clear();
		mRenderer.drawBox(d->value());
	}
private:
	BoxRenderer mRenderer;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::BoxVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::BoxVectorVisualization3D, mira::Visualization3D);
