/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVectorVisualization.C
 *    Visualization of vectors of poses in 2D space.
 *
 * @author Christian Reuther
 * @date   2014/08/27
 */

#include <transform/Pose.h>

#include <serialization/Serialization.h>
#include <visualization/Visualization2D.h>

#include <QGraphicsScene>
#include <QGraphicsPathItem>
#include <QVector>

#include <Eigen/Eigen>
#include <Eigen/Eigenvalues>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename Pose>
class PoseVectorVisualization2D :  public Visualization2D {
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	typedef typename PoseVectorTrait<Pose>::VectorType PoseVector;
	typedef QPair<QGraphicsLineItem*, QGraphicsLineItem*> LineItemPair;

public:
	PoseVectorVisualization2D() : mArrowLength(0.05f), mShowYAxis(false), mItem(NULL) {
		mDataChannel.setDataChangedCallback(boost::bind(&PoseVectorVisualization2D::onDataChanged, this, _1));
	}

	virtual ~PoseVectorVisualization2D() {
		auto site = this->getSite();
		// For some reason it is possible that the view gets destructed and re-constructed
		// when miracenter starts, so we need to make sure that we can access the items
		if(!mItem || !site || !site->getSceneManager())
			return;

		site->getSceneManager()->removeItem(mItem);
		foreach(LineItemPair items, mLineItems) {
			mItem->removeFromGroup(items.first);
			mItem->removeFromGroup(items.second);
			delete items.first;
			delete items.second;
		}
		delete mItem;
	}

	template <typename Reflector> void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Visualization2D);
		channelProperty(r, "PoseVector", mDataChannel, "Pose vector channel to be visualized");

		r.property("Length", mArrowLength, setter<float>(&PoseVectorVisualization2D::setArrowLength, this),
			"The length of the axes in [m]", 0.05f);
		r.property("Show Y axis", mShowYAxis, setter<bool>(&PoseVectorVisualization2D::setShowYAxis, this),
			"Whether or not to display the (quite redundant) y axis of the 2D pose", false);
	}

public:
	void setArrowLength(float length) {
		mArrowLength = length;
	}

	void setShowYAxis(bool show) {
		mShowYAxis = show;
		foreach(LineItemPair items, mLineItems)
			items.second->setVisible(mShowYAxis);
	}

public:
	virtual Visualization::DataConnection getDataConnection() {
		return Visualization::DataConnection(mDataChannel);
	}

	virtual void setEnabled(bool enabled) {
		Visualization2D::setEnabled(enabled);
		mItem->setVisible(enabled);
	}

	virtual void setupScene(IVisualization2DSite* site) {
		QGraphicsScene* mgr = site->getSceneManager();
		mPenXAxis.setColor(Qt::red);
		mPenXAxis.setCosmetic(true);
		mPenYAxis.setColor(Qt::green);
		mPenYAxis.setCosmetic(true);

		mItem = new QGraphicsItemGroup();
		mgr->addItem(mItem);
	}

	virtual QGraphicsItem* getItem()
	{
		return mItem;
	}

	void onDataChanged(ChannelRead<PoseVector> data) {
		const std::size_t newsize = data->value().size();
		const std::size_t oldsize = static_cast<std::size_t>(mLineItems.size());

		// First we see whether we need to resize to accomodate more objects
		if(newsize > oldsize) {
			for(std::size_t i = oldsize; i < newsize; ++i) {
				QGraphicsLineItem* x = new QGraphicsLineItem();
				QGraphicsLineItem* y = new QGraphicsLineItem();
				x->setPen(mPenXAxis);
				y->setPen(mPenYAxis);

				// Qt docu says: "QGraphicsItemGroup::addToGroup add the given
				// item to this group. The item will be reparented to this
				// group, but its position and transformation relative to the
				// scene will stay intact."
				// Therefore, me must flip the y-axis again here.

				mItem->addToGroup(x);
				mItem->addToGroup(y);
				mLineItems.append(LineItemPair(x, y));
			}
		} else if(oldsize > 0) {
			// Hide all items that are not needed anymore
			for(std::size_t i = oldsize - 1; i > newsize; --i) {
				const LineItemPair& pair = mLineItems.at(static_cast<int>(i));
				pair.first->setVisible(false);
				pair.second->setVisible(false);
			}
		}

		// Now update the item's positions
		for(std::size_t i = 0; i < newsize; ++i) {
			const LineItemPair& pair = mLineItems.at(static_cast<int>(i));
			setPose(pair, data->value().at(i));
		}

		// Ok, now we can transform our group item by the camera to data frame
		Pose2 cameraTransform = this->getAuthority().template getTransform<Pose2>(data->frameID, data->timestamp,
				this->getSite()->getCameraFrame(), data->timestamp, this->getSite()->getFixedFrame());
		mItem->setPos(static_cast<double>(cameraTransform.x()), static_cast<double>(cameraTransform.y()));
		mItem->setRotation(rad2deg<float>(cameraTransform.phi()));
	}

protected:
	void setPose(const LineItemPair& pair, Pose pose) {
		const float ax = pose.phi();
		const float ay = pose.phi() + pi<float>() * 0.5f;
		const float r = mArrowLength;

		// Calculate our offset points with the length
		const QPointF o(pose.x(), pose.y());
		const QPointF x(o.x() + r * cosf(ax), o.y() + r * sinf(ax));
		const QPointF y(o.x() + r * cosf(ay), o.y() + r * sinf(ay));

		pair.first->setLine(QLineF(o, x));
		pair.second->setLine(QLineF(o, y));

		pair.first->setVisible(true);
		pair.second->setVisible(mShowYAxis);
	}

protected:
	ChannelProperty<PoseVector> mDataChannel;
	float mArrowLength;
	bool mShowYAxis;

	QGraphicsItemGroup* mItem;
	QVector<LineItemPair> mLineItems;

	QPen mPenXAxis;
	QPen mPenYAxis;
};

///////////////////////////////////////////////////////////////////////////////

class Pose2VectorVisualization2D : public PoseVectorVisualization2D<Pose2>
{
	MIRA_META_OBJECT(Pose2VectorVisualization2D,
		("Name", "Pose2 Vector")
		("Description", "Visualization of 2D pose vectors")
		("Category", "Pose"))
};

class PoseCov2VectorVisualization2D : public PoseVectorVisualization2D<PoseCov2>
{
	MIRA_META_OBJECT(PoseCov2VectorVisualization2D,
		("Name", "PoseCov2 Vector")
		("Description", "Visualization of 2D pose vectors with covariance")
		("Category", "Pose"))
};

class Pose2dVectorVisualization2D : public PoseVectorVisualization2D<RigidTransform<double, 2>>
{
	MIRA_META_OBJECT(Pose2dVectorVisualization2D,
		("Name", "Pose2d Vector")
		("Description", "Visualization of 2D double pose vectors")
		("Category", "Pose"))
};

class PoseCov2dVectorVisualization2D : public PoseVectorVisualization2D<RigidTransformCov<double, 2>>
{
	MIRA_META_OBJECT(PoseCov2dVectorVisualization2D,
		("Name", "PoseCov2d Vector")
		("Description", "Visualization of 2D double pose vectors with covariance")
		("Category", "Pose"))
};

//////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::Pose2VectorVisualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov2VectorVisualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2dVectorVisualization2D, mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov2dVectorVisualization2D, mira::Visualization2D);
