#!/bin/bash

###############################################################################

if [ ! -f mira.root ]; then
	echo "ERROR: This script must run in the MIRA root dir."
	exit -1
fi

###############################################################################

DATESTR=`date +%Y-%m-%d`
WHATSNEW=whatsnew_${DATESTR}.txt

rm -f $WHATSNEW

###############################################################################

# Usage: getChangelog PathToPackageFile
#    This function checks the head of the changelog. If the changelog
#    does not contain a release date, the new changelog will be copied to
#    $WHATSNEW and the current date will be set a the release date.
function getChangelog {
	c=${1/.package/.changelog}
	
	# If the first line contains a release date, we can skip this.
	if [[ ! -z `head -n1 $c | grep -i release` ]] ; then
		echo "Skipping $1"
		return
	fi
		
	# Get version from head of changelog
	VERSION_CL=`head -n1 $c | \
		sed -r "s/Version[ \t]*([0-9]*)\.([0-9]*)\.([0-9]*).*/\1.\2.\3/g"`
	
	# Get version form package file
	VERSION_PKG=`cat $1 | grep MIRA_VERSION | \
		sed -r "s/MIRA_VERSION\([ \t]*([0-9]*)[ \t]*([0-9]*)[ \t]*([0-9]*)\)/\1.\2.\3/g"`

	# Verify, that boths versions are equal
	if [[ "$VERSION_CL" != "$VERSION_PKG" ]] ; then
		echo "ERROR: Version mismatch in package $1"
		exit -1
	fi
	
	echo "Processing $1"
	
	# Extract changelog
	# https://unix.stackexchange.com/questions/11305/show-all-the-file-up-to-the-match
	sed '/Release/Q' $c | tail -n +3 > $c.new

	# New output	
	name=$(echo "`basename $1`" | cut -f 1 -d '.')

	echo $name-$VERSION_PKG":" >> $WHATSNEW
	cat $c.new >> $WHATSNEW
	rm $c.new

	# Put release date into changelog
	cat $c | \
		sed "0,/Version $VERSION_PKG.*/s//Version $VERSION_PKG (Release date: $DATESTR)/" \
		> $c.new
	mv $c.new $c
}

###############################################################################

getChangelog make/dependencies/external.package
getChangelog MIRAenvironment.package
getChangelog base/MIRABase.package
getChangelog framework/MIRAFramework.package

for f in `find gui -name "*.package"`; do
	getChangelog $f
done

for f in `find tools -name "*.package"`; do
	getChangelog $f
done

for f in `find toolboxes -name "*.package"`; do
	getChangelog $f
done

echo "Wrote changelog summary to $WHATSNEW"

