###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Provides MIRA_DIST_LIBRARY, MIRA_DIST_BINARY, MIRA_DIST_FILE and MIRA_DIST_FILES
# a macro that install the created library, binary or file(s) to the <somewhere>.
# On Linux the library/binary is installed as a symbolic link.
#
# Usage:
#   MIRA_DIST_LIBRARY(TARGET [subdir])
#   MIRA_DIST_BINARY(TARGET [subdir])
#   MIRA_DIST_FILE(TARGET source [subdir])
#   MIRA_DIST_FILES(TARGET
#                   [FILES   <file1 ... fileN>]
#                   [SUBDIR  <subdir>]
#                  )
#
# The possible subdirs are defined in ALLOWED_LIB_SUBDIRS and ALLOWED_BIN_SUBDIRS.
# - For libs they are as follows: units loaders plugins gui ORGE
# - For bin they are as follows: tools
#
# Author: Erik Einhorn, Christian Martin
#
###############################################################################

# Generic macro for creating a symbolic link from sourcePath to destPath.
# The commands for doing so are assigned to the 
# specified target. Hence, the symbolic link/copy is created when the target
# is built. On Linux, the symbolic link even is created, if the target is up
# to date (this is important for smooth transition between debug and release 
# builds)
# This macro is used by MIRA_ADD_LIBRARY, MIRA_ADD_BINARY, ADD_MANIFEST, ...
#
macro(MIRA_DIST_FILE_EX target sourcePath destPath)

	# sourcePath and/or destPath may contain cmake generator expressions

	# extract directory name from destPath for directory creation
	get_filename_component(destDir ${destPath} PATH)

	# for debugging:
	#message("MIRA_DIST_FILE_EX")
	#message("    target     = ${target}")
	#message("    sourcePath = ${sourcePath}")
	#message("    destPath   = ${destPath}")
	#message("    destDir    = ${destDir}")

	# sanity check ${destDir} must not contain a generator expression
	string(GENEX_STRIP "${destDir}" no_genex)
	if(NOT "${destDir}" STREQUAL "${no_genex}")
		message(FATAL_ERROR "Can't use MIRA_DIST_FILE_EX with a generator "
		                    "expression within the destination path:\n"
		                    "  destPath = ${destPath}\n"
		                    "  destDir  = ${destDir}")
	endif()

	# Now add commands for 'installing' the file:
	#  - Create a separate xyz_dist target, which will trigger the commands
	#    (this ensures that we can switch seamlessly between release and debug
	#     builds, and it ensures that the files/links are refreshed even if the
	#     target is up to date)
	#  - Use symbolic links

	# ensure, that target_dist exists
	if(NOT TARGET ${target}_dist)
		add_custom_target(${target}_dist ALL)
		add_dependencies(${target}_dist ${target})
	endif()

	# add symlink creation as a custom command
	add_custom_command(TARGET ${target}_dist
		COMMAND ${CMAKE_COMMAND} -E make_directory "${destDir}"
		COMMAND ${CMAKE_COMMAND} -E remove "${destPath}"
		COMMAND ${CMAKE_COMMAND} -E create_symlink "${sourcePath}" "${destPath}"
	)

endmacro(MIRA_DIST_FILE_EX)

##############################################################################

# the allowed subdirs
set(ALLOWED_LIB_SUBDIRS units loaders plugins gui OGRE)

# the root lib dir (for .lib files)
set(LIB_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR}/lib)

###############################################################################

macro(MIRA_DIST_LIBRARY_EX target targetPath)
	set(subdir ${ARGV2}) # optional 2nd parameter that specifies the subdirectory

	# check if 'subdir' is in the list of allowed subdirectories (but only if we have a subdir)
	if(${ARGC} GREATER 2)
		list(FIND ALLOWED_LIB_SUBDIRS ${subdir} found)
		if(${found} EQUAL -1)
			message("found = " ${found})
			message(FATAL_ERROR "The following subdirectories are allowed for MIRA_DIST_LIBRARY only: " "${ALLOWED_LIB_SUBDIRS}")
		endif()
	endif()

	# get values of VERSION and SOVERSION
	get_target_property(targetVersion ${target} VERSION)
	get_target_property(targetSOVersion ${target} SOVERSION)
	
	# library target directory
	set(libDistDir "${LIB_DIST_ROOT_DIR}/${subdir}")

	# for debugging:
	#message("MIRA_DIST_LIBRARY_EX")
	#message("    subdir          = " ${subdir})
	#message("    libDistDir      = " ${libDistDir})
	#message("    targetPath      = " ${targetPath})
	#message("    targetVersion   = " ${targetVersion})
	#message("    targetSOVersion = " ${targetSOVersion})

	# Ensure, that either both of the version target properties VERSION
	# and SOVERSION is used or none.
	if ((NOT ${targetSOVersion} MATCHES NOTFOUND) AND (${targetVersion} MATCHES NOTFOUND))
		message(FATAL_ERROR "The target " ${target} " uses only one property SOVERSION or VERSION! Always use both or none.")
	endif()


	if ((${targetVersion} MATCHES NOTFOUND) AND (${targetSOVersion} MATCHES NOTFOUND))
		# Both properties VERSION and SOVERSION are not used.
		# Only create a symbolic link for the target
		
		string(GENEX_STRIP "${targetPath}" no_genex)
		if("${targetPath}" STREQUAL "${no_genex}")
			# The targetPath doesn't contain a generator expression.
			# Therefore, we can use get_filename_component

			get_filename_component(targetFilename ${targetPath} NAME)
			MIRA_DIST_FILE_EX(${target} ${targetPath} ${libDistDir}/${targetFilename})
		else()
			MIRA_DIST_FILE_EX(${target} ${targetPath} ${libDistDir}/$<TARGET_FILE_NAME:${target}>)
		endif()

	else()
		# Both properties VERSION and SOVERSION are used. So create
		# a symbolic links for all three types.
		
		# In the following command, the "lib" and "so" also could be replaced
		# by using the corresponding cmake generator. But this would make the
		# commands almost unreadable.

		# lib/libFooBar.so.0.1.2 -> build/..../libFooBar.so.0.1.2
		MIRA_DIST_FILE_EX(${target}
			${targetPath}
			${libDistDir}/$<TARGET_FILE_NAME:${target}>)
		
		# lib/libFooBar.so.0.1   -> libFooBar.so.0.1.2
		MIRA_DIST_FILE_EX(${target}
			$<TARGET_FILE_NAME:${target}>
			${libDistDir}/lib$<TARGET_FILE_BASE_NAME:${target}>.so.${targetSOVersion})

		# lib/libFooBar.so       -> libFooBar.so.0.1
		MIRA_DIST_FILE_EX(${target}
			lib$<TARGET_FILE_BASE_NAME:${target}>.so.${targetSOVersion}
			${libDistDir}/lib$<TARGET_FILE_BASE_NAME:${target}>.so)
						
	endif()

endmacro(MIRA_DIST_LIBRARY_EX)

###############################################################################

macro(MIRA_DIST_LIBRARY target)
	set(subdir ${ARGV1}) # optional 2nd parameter that specifies the subdirectory

	# for debugging:
	#message("MIRA_DIST_LIBRARY")
	#message("    subdir     = ${subdir}")
	#message("    targetPath = ${targetPath}")

	MIRA_DIST_LIBRARY_EX(${target} $<TARGET_FILE:${target}> ${subdir})

endmacro(MIRA_DIST_LIBRARY)

###############################################################################

# the allowed subdirs
set(ALLOWED_BIN_SUBDIRS tools)

# the root bin dir
set(BIN_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR}/bin)

macro(MIRA_DIST_BINARY target)
	set(subdir ${ARGV1}) # optional 2nd parameter that specifies the subdirectory

	# check if 'subdir' is in the list of allowed subdirectories (but only if we have a subdir)
	if(${ARGC} GREATER 1)
		list(FIND ALLOWED_BIN_SUBDIRS ${subdir} found)
		if(${found} EQUAL -1)
			message("found = " ${found})
			message(FATAL_ERROR "The following subdirectories are allowed for MIRA_DIST_BINARY only: " "${ALLOWED_BIN_SUBDIRS}")
		endif(${found} EQUAL -1)
	endif(${ARGC} GREATER 1)

	# build the dir and full path where we should put the symbolic link
	set(installDir  ${BIN_DIST_ROOT_DIR}/${subdir})

	# for debugging:
	#message("MIRA_DIST_BINARY")
	#message("    subdir     = " ${subdir})
	#message("    installDir = " ${installDir})

	MIRA_DIST_FILE_EX(${target} $<TARGET_FILE:${target}> ${installDir}/$<TARGET_FILE_NAME:${target}>)
	
endmacro(MIRA_DIST_BINARY)

###############################################################################

# the root bin dir
set(FILES_DIST_ROOT_DIR ${CMAKE_SOURCE_DIR})

macro(MIRA_DIST_FILE target source)
	set(subdir ${ARGV2}) # optional 2nd parameter that specifies the subdirectory

	# get the full path to the source file
	set(fullSourcePath ${CMAKE_CURRENT_SOURCE_DIR}/${source})

	# extract the source filename
	get_filename_component(sourceFilename ${fullSourcePath} NAME)

	# build the dir name and full path name of the destination
	set(installDir  ${FILES_DIST_ROOT_DIR}/${subdir})
	set(installPath ${installDir}/${sourceFilename})

	# for debugging:
	#message("subdir         = " ${subdir})
	#message("fullSourcePath = " ${fullSourcePath})
	#message("sourceFilename = " ${sourceFilename})
	#message("installDir     = " ${installDir})
	#message("installPath    = " ${installPath})

	MIRA_DIST_FILE_EX(${target} ${fullSourcePath} ${installPath})

endmacro(MIRA_DIST_FILE)

###############################################################################

macro(MIRA_DIST_FILES target)
	###########################################################################
	# Parse the macro arguments

	CMAKE_PARSE_ARGUMENTS(arg
		# options:
		""
		# one-value arguments:
		"SUBDIR"
		# multi-value arguments:
		"FILES"
		${ARGN})

	# Ensure, that we have at least one source file
	if (NOT arg_FILES)
		message(FATAL_ERROR "Target ${target} has no files to install.")
	endif()

	if (NOT arg_SUBDIR)
		foreach(item ${arg_FILES})
			MIRA_DIST_FILE(${target} ${item})
		endforeach(item)
	else()
		foreach(item ${arg_FILES})
			MIRA_DIST_FILE(${target} ${item} ${arg_SUBDIR})
		endforeach(item)
	endif()
endmacro(MIRA_DIST_FILES)

###############################################################################

