/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RectTool.C
 *    Implementation of RectTool.
 *
 * @author Tim Langner
 * @date   2011/08/18
 */

#include <QMouseEvent>

#include <visualization/3d/RectTool.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void RectTool3D::onMousePressed(QMouseEvent* e)
{
	if (e->button() != Qt::LeftButton)
		return;

	auto i = getIntersection(e->pos().x(), e->pos().y());
	if (!i.first.empty())
	{
		mSelectedObject = i.first;
		mRect.minCorner.x() = mRect.maxCorner.x() = i.second.x;
		mRect.minCorner.y() = mRect.maxCorner.y() = i.second.y;
		mRectObject->setRect(mRect);
	}
}

void RectTool3D::onMouseReleased(QMouseEvent* e)
{
	if (e->button() != Qt::LeftButton)
		return;
	
	auto i = getIntersection(e->pos().x(), e->pos().y(), mSelectedObject);
	if (!i.first.empty())
	{
		mRect.maxCorner.x() = i.second.x;
		mRect.maxCorner.y() = i.second.y;
		mRectObject->setRect(mRect);
		onNewRect(mRect);
	}
	mSelectedObject = "";
}

void RectTool3D::onMouseMoved(QMouseEvent* e)
{
	if (!mSelectedObject.empty())
	{
		auto i = getIntersection(e->pos().x(), e->pos().y(), mSelectedObject);
		if (i.first.empty())
			return;
		mRect.maxCorner.x() = i.second.x;
		mRect.maxCorner.y() = i.second.y;
		mRectObject->setRect(mRect);
	}
}

/// called when the user has activated the tool.
void RectTool3D::activate()
{
	mRectObject->setVisible(true);
}

/// called when the user has deactivated the tool by selecting another one.
void RectTool3D::deactivate()
{
	mRectObject->setVisible(false);
}

std::pair<std::string, Ogre::Vector3> RectTool3D::getIntersection(float x, float y,
                                                                  const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject")
                        : SceneQueryTool::getIntersection(x, y, mask, true);
}

void RectTool3D::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mRectObject.reset(new RectObject(site->getSceneManager()));
	mRectObject->setVisible(false);
	mSelectedObject = "";
}

///////////////////////////////////////////////////////////////////////////////

}
