/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CovObject.C
 *   Visualization of covariance.
 *
 * @author Erik Einhorn
 * @date   08/21/2011
 */

#include <visualization/3d/CovObject.h>

#include <OGRE/OgreSceneNode.h>

#include <opencv2/core/core.hpp>
#include <opencv2/core/eigen.hpp>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

CovObject::CovObject(Ogre::SceneManager* sceneManager,
                     Ogre::SceneNode* parent) :
	MeshObject("Sphere.mesh", sceneManager, parent)
{
	mW = Eigen::Vector3f::Zero();
	mSigma = 1.0f;
}

void CovObject::setSigma(float sigma)
{
	mSigma = sigma;
	// scale the sphere according to our covariances along the axes
	// (contained in the singular vals)

	if(boost::math::isnan(sigma)||
	   boost::math::isnan(mW(0))||
	   boost::math::isnan(mW(1))||
	   boost::math::isnan(mW(2)))
		return;

	getNode()->setScale(sqrt(mW(0)) * mSigma,
	                    sqrt(mW(1)) * mSigma,
	                    sqrt(mW(2)) * mSigma);
}

void CovObject::setMean(const Eigen::Vector2f& x)
{
	Eigen::Vector3f x3;
	x3 << x,0.0f;
	setMean(x3);
}

void CovObject::setMean(const Eigen::Vector3f& x)
{
	setPosition(x);
}

void CovObject::setCovariance(const Eigen::Matrix2f& cov)
{
	// zero cov causes failing assert(U.determinant()>0) in setCovariance(const Eigen::Matrix3f& cov)
	// (svd seems wrong??)
	if(cov.trace() < std::numeric_limits<float>::epsilon())
	{
		mW << 0.0f, 0.0f, 0.0f;
		return;
	}

	// build a 3d cov.matrix with epsilon z
	const float eps = 1.0e-4f;
	Eigen::Matrix3f cov3;
	cov3.setZero();
	cov3.block<2,2>(0,0) = cov.block<2,2>(0,0);
	cov3(2,2) = eps;
	setCovariance(cov3);
}

void CovObject::setCovariance(const Eigen::Matrix3f& cov)
{
/*
 	THIS SOMETIMES CRASHES, SO USE OPENCV BELOW!!!
	Eigen::JacobiSVD<Eigen::Matrix3f> svd(cov, Eigen::ComputeThinU | Eigen::ComputeThinV);

	mW = svd.singularValues();
	setSigma(mSigma); // will set the scale

	// V contains the rotation matrix of our ellipsoid
	Eigen::Quaternionf q(svd.matrixV());
	//std::cout << "V: " << svd.matrixV() << std::endl;
	//std::cout << "q: " << q.w() << ", " << q.x() << ", " << q.y() << ", " << q.z() << std::endl;
	getNode()->setOrientation(q.w(),q.x(),q.y(),q.z());
*/
	// if we have a zero - cov matrix, escape here
	if(cov.trace() < std::numeric_limits<float>::epsilon())
	{
		mW << 0.0f, 0.0f, 0.0f;
		return;
	}

	cv::Mat cvCov;
	cv::eigen2cv(cov,cvCov);
	cv::SVD svd(cvCov, cv::SVD::FULL_UV);

	cv::cv2eigen(svd.w, mW);
	Eigen::Matrix3f U;
	cv::cv2eigen(svd.u, U);

	// if there is a reflection within the U matrix, then mirror Z axis
	// in order to obtain a valid rotation matrix
	if(U.determinant()<0)
		U.col(2) = -U.col(2);

	assert(U.determinant()>0);

	Eigen::Quaternionf q(U);
	getNode()->setOrientation(q.w(),q.x(),q.y(),q.z());

	setSigma(mSigma);
}

void CovObject::setCovariance(const Eigen::Vector3f& sv,
                              const Ogre::Quaternion& orientation)
{
	mW = sv;
	getNode()->setOrientation(orientation);
	setSigma(mSigma);
}

///////////////////////////////////////////////////////////////////////////////

}
