/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PageViewPart.C
 *    Implementation of PageViewPart.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/30
 */

#include <rcp/PageViewPart.h>

#include <QStackedWidget>

#include <serialization/Serialization.h>

#include <rcp/ViewPartTitleBar.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PageViewPart::PageViewPart() : mContainer(NULL)
{
	mLock = new QToolButton(this);
	QIcon lockIcon;
	lockIcon.addPixmap(QPixmap(":/icons/Unlocked.png"), QIcon::Normal, QIcon::Off);
	lockIcon.addPixmap(QPixmap(":/icons/Locked.png"), QIcon::Normal, QIcon::On);

	mLock->setIcon(lockIcon);
	mLock->setToolTip(tr("Lock current content"));
	mLock->setFixedSize(18, 18);
	mLock->setAutoRaise(true);
	mLock->setCheckable(true);

	getTitleBar()->insertRightButton(mLock);
}

PageViewPart::~PageViewPart()
{
	getSite()->removePartListener(this);

	// prevent our pages from destruction and disconnect them from us
	while(mContainer->count()>0) {
		QWidget* w = mContainer->widget(0);
		mContainer->removeWidget(w);
		w->setParent(NULL);
	}
}

QWidget* PageViewPart::createPartControl()
{
	mContainer = new QStackedWidget(this);

	mOriginalTitle = this->windowTitle();

	getSite()->addPartListener(this);
	setPage(createDefaultPage());

	return mContainer;
}

void PageViewPart::editorActivated(IWorkbenchPart* part)
{
	if(mLock->isChecked()) // do nothing if the content is locked
		return;

	QWidget* page = createPage(part);
	if(page==NULL) // do nothing, if derived class has returned NULL to indicate no change
		return;

	setPage(page);
}

void PageViewPart::editorClosed(IWorkbenchPart* part)
{
	// nothing to do here
}

void PageViewPart::viewActivated(IWorkbenchPart* part)
{
	if(mLock->isChecked()) // do nothing if the content is locked
		return;

	QWidget* page = createPage(part);
	if(page==NULL) // do nothing, if derived class has returned NULL to indicate no change
		return;

	setPage(page);
}

void PageViewPart::viewClosed(IWorkbenchPart* part)
{
	// nothing to do here
}

void PageViewPart::setPage(QWidget* page)
{
	assert(page!=NULL);

	// generate and set title
	QString title = mOriginalTitle;
	QString subTitle = page->windowTitle();
	if(!subTitle.isEmpty())
		title += " [" + subTitle + "]";
	setWindowTitle(title);
	assert(mTitleBar!=NULL);
	mTitleBar->update();

	// check if that page was already added
	for(int i=0; i<mContainer->count(); ++i)
	{
		QWidget* w = mContainer->widget(i);
		if(w==page) {
			// found, so show it
			mContainer->setCurrentWidget(w);
			return;
		}
	}

	// page was not added yet, so do this now
	page->setParent(this);
	mContainer->addWidget(page);
	mContainer->setCurrentWidget(page);
}

//////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION( mira::PageViewPart, mira::ViewPart );
