/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTool.h
 *    Declaration of VisualizationTool.
 *
 * @author Erik Einhorn
 * @date   2011/07/01
 */

#include <QtGui/QIcon>

#include <visualization/Visualization.h>


#ifndef _MIRA_VISUALIZATIONTOOL_H_
#define _MIRA_VISUALIZATIONTOOL_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class VisualizationTool
	: public Object, public IAuthorityProvider, public Authority
{
	MIRA_ABSTRACT_OBJECT(VisualizationTool)
public:

	VisualizationTool();
	virtual ~VisualizationTool() {};

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.itemName(this->getClass().getMetaInfo("Name"));
	}

public:

	/**
	 * Is called by the visualization view when the tool is attached.
	 * The tool should initialize itself.
	 */
	virtual void init(IVisualizationSite* site);

	/**
	 * Returns a pointer to the IVisualizationSite interface that was set
	 * in the init() method.
	 * Or returns NULL, if init() was not yet called.
	 */
	virtual IVisualizationSite* getSite() = 0;

public:

	/**
	 * Each tool must return an icon that is displayed within the toolbar.
	 */
	virtual QIcon getIcon() = 0;


	/**
	 * A tool may return a shortcut key that temporarily activates the tool.
	 * If the key is pressed, the currently active tool is displaced and this
	 * tool is activated. If the key is released, the previously displaced
	 * tool is reactived. This method should return the Qt keycode, which
	 * may also contain modifier keys, e.g.
	 * (Qt::Key_Control)  or (Qt::CTRL | Qt::Key_F1), etc.
	 */
	virtual int getDisplaceShortCut() const { return 0; }



public:

	/// called when the user has activated the tool.
	virtual void activate() {}

	/// called when the user has deactivated the tool by selecting another one.
	virtual void deactivate() {}

public:

	/**
	 * Is called regularly by the VisualizationView if the tool is active to
	 * update the tool.
	 * The time span since the last call of update is passed as parameter.
	 * It calls update() that can be overwritten in derived classes
	 */
	virtual void processUpdate(Duration dt) { update(dt); }

protected:
	/**
	 * This method can be implemented in the derived classes.
	 * See processUpdate()
	 */
	virtual void update(Duration dt) {
		onUpdate(dt);		// Only for backward compatibility
	}

	/**
	 * \deprecated. Please overwrite update()
	 * Only provided for backward compatibility
	 */
	virtual void onUpdate(Duration dt) {}

public:

	/// Called if the tool is active and a mouse button was pressed.
	virtual void onMousePressed(QMouseEvent* e) {}
	virtual void onMousePressed(QMouseEvent* e, float devicePixelRatio) { onMousePressed(e); }
	/// Called if the tool is active and a mouse button was released.
	virtual void onMouseReleased(QMouseEvent* e) {}
	virtual void onMouseReleased(QMouseEvent* e, float devicePixelRatio) { onMouseReleased(e); }
	/// Called if the tool is active and a double click as occurred.
	virtual void onMouseDoubleClick(QMouseEvent* e) {}
	virtual void onMouseDoubleClick(QMouseEvent* e, float devicePixelRatio) { onMouseDoubleClick(e); }
	/// Called if the tool is active and the mouse was moved.
	virtual void onMouseMoved(QMouseEvent* e) {}
	virtual void onMouseMoved(QMouseEvent* e, float devicePixelRatio) { onMouseMoved(e); }
	/// Called if the tool is active and the mouse wheel was moved.
	virtual void onMouseWheel(QWheelEvent * e) {}
	virtual void onMouseWheel(QWheelEvent * e, float devicePixelRatio) { onMouseWheel(e); }

protected:

	void updateToolTip(QMouseEvent* e, const QString& text);

public: // implementation of IAuthorityProvider

	virtual Authority& getAuthority();

protected:

	QString mLastToolTip;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
