/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization2DContainer.h
 *    QWidget containing a Visualization2DSite but without being a RCP part.
 *
 * @author Christian Reuther
 * @date   2014/09/08
 */

#ifndef _MIRA_VISUALIZATION2DCONTAINER_H_
#define _MIRA_VISUALIZATION2DCONTAINER_H_

#include <visualization/Visualization2D.h>

#include <QEvent>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QtGui/QMouseEvent>

#include <visualization/VisualizationTool2D.h>

namespace mira {

class PropertyEditor;

///////////////////////////////////////////////////////////////////////////////

class Visualization2DContainer : public QGraphicsView, public IVisualization2DSite
{
	Q_OBJECT
public:
	Visualization2DContainer(QWidget* parent = NULL, bool enableOrbitTool = true);
	virtual ~Visualization2DContainer();

	Visualization2D* addVisualization(const std::string& visualizationClassName);
	void addVisualization(Visualization2D* vis);

	void removeVisualization(Visualization2D* vis);

public:
	virtual QGraphicsScene* getSceneManager();
	virtual QGraphicsView* getViewManager();

	virtual const Camera& getCamera() const;
	virtual void setCamera(const Camera& camera);

	virtual float getToolBaseZValue() const;

	virtual const std::string& getFixedFrame() const;
	virtual const std::string& getCameraFrame() const;

	virtual void setFixedFrame(const std::string& frame);
	virtual void setCameraFrame(const std::string& frame);

	virtual void setBackgroundColor(const QColor& color);
	virtual void fitRectInView(const QRectF& rect);

public: // Access properties of visualizations
	std::string getPropertyString(Visualization2D* vis, const std::string& property);
	template <typename T> T getProperty(Visualization2D* vis, const std::string& property) {
		return fromString<T>(getPropertyString(vis,property));
	}

	void setPropertyString(Visualization2D* vis, const std::string& property, const std::string& value);
	template <typename T> void setProperty(Visualization2D* vis, const std::string& property, const T& value) {
		setPropertyString(vis,property,toString<T>(value));
	}

	// Create an editor widget (hidden, parentless) or return the existing one
	PropertyEditor* getPropertyEditor(QWidget* parent = NULL);

protected: // Interface for derived classes
	virtual void processUpdate(Duration d) {}

protected:
	virtual void mousePressEvent(QMouseEvent* event);
	virtual void mouseMoveEvent(QMouseEvent* event);
	virtual void mouseReleaseEvent(QMouseEvent* event);
	virtual void wheelEvent (QWheelEvent* event);
	virtual void timerEvent(QTimerEvent* e);

protected:
	virtual void updateCamera();
	virtual void setupVisualizations(bool enableOrbitTool = true);

	Camera mCameraSettings;

	QGraphicsScene* mScene;
	QColor mBackgroundColor;

	boost::shared_ptr<VisualizationTool2D> mActiveTool;

	Time mLastUpdateTime;

	boost::shared_ptr<PropertyNode> mGeneralProperties;
	std::map<Visualization2D*, boost::shared_ptr<PropertyNode>> mVisualizations;

	PropertyEditor* mEditor;

	std::string mFixedFrame;
	std::string mCameraFrame;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
