/*
 * Copyright (C) 2016 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IndexSelectorProperty.h
 *    Property to select an element from a set type.
 *
 * @author Christof Schröter
 * @date   2016/12/12
 */

#ifndef _MIRA_INDEXSELECTORPROPERTY_H_
#define _MIRA_INDEXSELECTORPROPERTY_H_


#include <OGRE/OgreColourValue.h>

#include <serialization/GetterSetter.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Base for template class IndexSelectorProperty
class IndexSelectorPropertyBase
{
public:
	IndexSelectorPropertyBase(boost::function<void()> callback)
		: mIndex(0), mWrapIndex(INT_MAX/2), mNotifyChange(callback) {}

public:
	int index() { return mIndex; }

public:
	/**
	 * Dynamically set the first invalid index, usually according
	 * to the visualized data itself (once it is known).
	 * Adapts the currently selected index, but does not trigger a redraw itself,
	 * so it is safe to call this from within the drawing method,
	 * then use the (wrapped) index for actual drawing.
	 */
	void setWrapIndex(unsigned int wrap);

protected:
	void wrapIndex();
	void setIndex(int index);

protected:
	/*
	 * A valid index should be of unsigned int type actually,
	 * but by declaring it signed we can take negative values from the property
	 * editor and wrap them around the maximum valid index
	 * (dynamic, so unknown to the editor) internally.
	 */
	int mIndex;
	int mWrapIndex;
	boost::function< void()> mNotifyChange;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A special property class that can be used to provide a mechanism for selecting an
 * element by index from data representing a set of similar objects.
 *
 * The property is initialized with a callback function, which is called on any
 * property change (thus allowing instant redraw).
 *
 * @tparam Transparent Controls the appearance in the property editor. If true,
 *                     an editor spinbox appears inline. If false, this property
 *                     creates an expandable node with an Index property.
 *                     Subclasses adding further properties must derive from the
 *                     Transparent=false specialization.
 */
template <bool Transparent = true>
class IndexSelectorProperty : public IndexSelectorPropertyBase
{
public:
	IndexSelectorProperty(boost::function<void()> callback)
		: IndexSelectorPropertyBase(callback) {}

	template <typename Class>
	IndexSelectorProperty(void (Class::*f)(), Class* obj)
		: IndexSelectorPropertyBase(boost::bind(f, obj)) {}

public:
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.delegate(mIndex,
		           setter(&IndexSelectorProperty::setIndex, (IndexSelectorPropertyBase*)this));
	}
};

///////////////////////////////////////////////////////////////////////////////

template <typename SerializerTag>
class IsTransparentSerializable<IndexSelectorProperty<true>, SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

/**
 * Explicit specialization for Transparent=false (different reflect() method).
 */
template <>
class IndexSelectorProperty<false> : public IndexSelectorPropertyBase
{
public:
	IndexSelectorProperty(boost::function<void()> callback)
		: IndexSelectorPropertyBase(callback) {}

	template <typename Class>
	IndexSelectorProperty(void (Class::*f)(), Class* obj)
		: IndexSelectorPropertyBase(boost::bind(f, obj)) {}

public:
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Index", mIndex,
		           setter(&IndexSelectorProperty::setIndex, (IndexSelectorPropertyBase*)this),
		           "The index of the selected element", 0);
	}
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
